<!DOCTYPE html>
<html lang="pt-BR">
<head>
  <meta charset="UTF-8">
  <title>Posição e Direção do Barco</title>
  <style>
    #map { height: 90vh; width: 100%; }
    body { margin: 0; font-family: Arial, sans-serif; }
    #info { padding: 10px; background: #f2f2f2; font-size: 16px; }
  </style>
</head>
<body>
  <div id="info">
    <strong>Direção estimada:</strong> <span id="direcao">Calculando...</span>
  </div>
  <div id="map"></div>

  <script src="https://cdnjs.cloudflare.com/ajax/libs/togeojson/0.16.0/togeojson.min.js"></script>
  <script src="https://cdnjs.cloudflare.com/ajax/libs/leaflet/1.9.3/leaflet.js"></script>
  <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/leaflet/1.9.3/leaflet.css" />

  <script>
    const pontoPartida = { lat: -3.811096, lon: -69.405915 }; // sua coordenada de partida
    const velocidade_kmph = 10;
    const velocidade_mps = velocidade_kmph * 1000 / 3600;
    let map, rota = [], marcador, linha;

    function carregarGPX() {
      fetch("solimoes.gpx")
        .then(res => res.text())
        .then(gpxText => {
          const parser = new DOMParser();
          const gpx = parser.parseFromString(gpxText, "application/xml");
          const trkpts = gpx.querySelectorAll("trkpt");

          rota = Array.from(trkpts).map(pt => ({
            lat: parseFloat(pt.getAttribute("lat")),
            lon: parseFloat(pt.getAttribute("lon"))
          }));

          iniciarMapa();
        })
        .catch(err => alert("Erro ao carregar o GPX: " + err));
    }

    function iniciarMapa() {
      map = L.map('map').setView([rota[0].lat, rota[0].lon], 13);
      L.tileLayer('https://{s}.tile.openstreetmap.org/{z}/{x}/{y}.png').addTo(map);

      const latlngs = rota.map(p => [p.lat, p.lon]);
      linha = L.polyline(latlngs, { color: 'blue' }).addTo(map);
      marcador = L.marker(latlngs[0]).addTo(map);

      atualizarPosicao();
      setInterval(atualizarPosicao, 10000);
    }

    function atualizarPosicao() {
      const agora = new Date();
      const horaPartida = new Date();
      horaPartida.setHours(6, 0, 0, 0);

      let segundos = (agora - horaPartida) / 1000;
      if (segundos < 0) segundos = 0;
      const distanciaPercorrida = segundos * velocidade_mps;

      let total = 0;
      for (let i = 0; i < rota.length - 1; i++) {
        const p1 = rota[i], p2 = rota[i + 1];
        const dist = haversine(p1.lat, p1.lon, p2.lat, p2.lon);
        if (total + dist >= distanciaPercorrida) {
          const frac = (distanciaPercorrida - total) / dist;
          const lat = p1.lat + (p2.lat - p1.lat) * frac;
          const lon = p1.lon + (p2.lon - p1.lon) * frac;
          marcador.setLatLng([lat, lon]);
          atualizarDirecao(pontoPartida.lat, pontoPartida.lon, lat, lon);
          return;
        }
        total += dist;
      }
      marcador.setLatLng([rota[rota.length - 1].lat, rota[rota.length - 1].lon]);
    }

    function atualizarDirecao(lat1, lon1, lat2, lon2) {
      const angulo = calcularRumo(lat1, lon1, lat2, lon2);
      const texto = direcaoTexto(angulo);
      document.getElementById("direcao").textContent = `${angulo.toFixed(2)}° (${texto})`;
    }

    function calcularRumo(lat1, lon1, lat2, lon2) {
      const toRad = deg => deg * Math.PI / 180;
      const toDeg = rad => rad * 180 / Math.PI;
      const dLon = toRad(lon2 - lon1);
      const y = Math.sin(dLon) * Math.cos(toRad(lat2));
      const x = Math.cos(toRad(lat1)) * Math.sin(toRad(lat2)) -
                Math.sin(toRad(lat1)) * Math.cos(toRad(lat2)) * Math.cos(dLon);
      const brng = Math.atan2(y, x);
      return (toDeg(brng) + 360) % 360;
    }

    function direcaoTexto(angulo) {
      const direcoes = ['Norte', 'Nordeste', 'Leste', 'Sudeste', 'Sul', 'Sudoeste', 'Oeste', 'Noroeste'];
      const index = Math.round(angulo / 45) % 8;
      return direcoes[index];
    }

    function haversine(lat1, lon1, lat2, lon2) {
      const R = 6371000;
      const toRad = deg => deg * Math.PI / 180;
      const dLat = toRad(lat2 - lat1);
      const dLon = toRad(lon2 - lon1);
      const a = Math.sin(dLat / 2) ** 2 +
                Math.cos(toRad(lat1)) * Math.cos(toRad(lat2)) * Math.sin(dLon / 2) ** 2;
      return R * 2 * Math.atan2(Math.sqrt(a), Math.sqrt(1 - a));
    }

    carregarGPX();
  </script>
</body>
</html>