<?php
/**
 * Plugin Name: Woo Avulso + Desconto FX (Gateways + Serviço/Gratuito)
 * Description: Gateways avulsos (Dinheiro, Pix, Débito, Crédito) com desconto do operador (fixo/%), modo gratuito (total=apenas serviço) e suporte a Checkout Clássico e Blocks.
 * Version: 1.0.0
 * Author: FX
 * License: GPLv2 or later
 * Requires Plugins: woocommerce
 */

if ( ! defined('ABSPATH') ) { exit; }

/* ------------------------------------------------------------------
 * Verificações básicas
 * ------------------------------------------------------------------*/
add_action('plugins_loaded', function(){
  if ( ! class_exists('WooCommerce') ) {
    add_action('admin_notices', function(){
      echo '<div class="notice notice-error"><p><strong>Woo Avulso + Desconto FX</strong> requer WooCommerce ativo.</p></div>';
    });
  }
});

/* -------------------------------------------------
 * Compatibilidade com Cart/Checkout Blocks
 * -------------------------------------------------*/
add_action('before_woocommerce_init', function () {
  if ( class_exists('\Automattic\WooCommerce\Utilities\FeaturesUtil') ) {
    \Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility(
      'cart_checkout_blocks', __FILE__, true
    );
  }
});

/* -------------------------------------------------
 * Helpers — parse, sessão do desconto e identificação de serviço
 * -------------------------------------------------*/
if ( ! function_exists('wa_fx_parse_number') ) {
  function wa_fx_parse_number($raw) {
    $raw = is_string($raw) ? $raw : '';
    $raw = preg_replace('/[^0-9\.,-]/', '', $raw);
    if (strpos($raw, ',') !== false) { $raw = str_replace('.', '', $raw); $raw = str_replace(',', '.', $raw); }
    $val = floatval($raw);
    return $val < 0 ? 0 : $val;
  }
}

if ( ! function_exists('wa_fx_set_discount_to_session') ) {
  function wa_fx_set_discount_to_session($type, $value, $gratis = null) {
    if ( ! function_exists('WC') || ! WC()->session ) return;
    $type  = in_array($type, ['fixed','percent'], true) ? $type : 'fixed';
    $value = max(0, floatval($value));
    WC()->session->set('wa_discount_type', $type);
    WC()->session->set('wa_discount', $value);
    if ($gratis !== null) {
      WC()->session->set('wa_mode_gratis', (bool)$gratis);
    }
  }
}

/* Identifica se um item do carrinho é “serviço” */
if ( ! function_exists('wa_fx_is_service_item') ) {
  function wa_fx_is_service_item($cart_item) {
    $product = isset($cart_item['data']) ? $cart_item['data'] : null;
    if ( ! $product || ! is_a($product, 'WC_Product') ) return false;

    // 1) Meta _wa_is_service = yes
    $meta_flag = $product->get_meta('_wa_is_service', true);
    if ( $meta_flag && strtolower($meta_flag) === 'yes' ) return true;

    // 2) SKU SERVICO
    $sku = $product->get_sku();
    if ( $sku && strtoupper($sku) === 'SERVICO' ) return true;

    // 3) Categoria slug 'servico'
    $terms = wc_get_product_terms($product->get_id(), 'product_cat', ['fields'=>'slugs']);
    if ( is_array($terms) && in_array('servico', $terms, true) ) return true;

    // Filtro para customização
    return (bool) apply_filters('wa_fx_is_service_item', false, $cart_item, $product);
  }
}

/* Soma subtotais por tipo */
if ( ! function_exists('wa_fx_split_subtotals') ) {
  function wa_fx_split_subtotals($cart) {
    $service = 0.0;
    $others  = 0.0;
    if ( ! $cart ) return [0.0, 0.0];
    foreach ($cart->get_cart() as $item) {
      $line = isset($item['line_subtotal']) ? floatval($item['line_subtotal']) : 0.0;
      if ($line <= 0) continue;
      if ( wa_fx_is_service_item($item) ) { $service += $line; }
      else { $others += $line; }
    }
    return [$service, $others];
  }
}

/* Sempre iniciar com 0,00 e sem gratuito ao abrir o checkout */
add_action('woocommerce_checkout_init', function(){
  wa_fx_set_discount_to_session('fixed', 0, false);
});

/* -------------------------------------------------
 * AJAX para setar desconto + modo gratuito
 * -------------------------------------------------*/
add_action('wp_ajax_wa_set_discount', 'wa_fx_ajax_set_discount');
add_action('wp_ajax_nopriv_wa_set_discount', 'wa_fx_ajax_set_discount');
function wa_fx_ajax_set_discount() {
  $type   = isset($_POST['type'])   ? sanitize_text_field(wp_unslash($_POST['type']))   : 'fixed';
  $value  = isset($_POST['value'])  ? wa_fx_parse_number(wp_unslash($_POST['value']))   : 0;
  $gratis = isset($_POST['gratis']) ? (bool) intval(wp_unslash($_POST['gratis']))       : null;
  wa_fx_set_discount_to_session($type, $value, $gratis);
  wp_send_json_success(['ok'=>true,'type'=>$type,'value'=>$value,'gratis'=>$gratis]);
}

/* -------------------------------------------------
 * FEE do desconto + Modo gratuito
 * -------------------------------------------------*/
add_action('woocommerce_cart_calculate_fees', function($cart){
  if ( is_admin() && ! defined('DOING_AJAX') ) return;
  if ( ! function_exists('WC') || ! WC()->cart ) return;
  if ( is_wc_endpoint_url('order-received') ) return;

  $session = WC()->session;
  $gratis  = $session ? (bool) $session->get('wa_mode_gratis', false) : false;
  $type    = $session ? $session->get('wa_discount_type', 'fixed') : 'fixed';
  $val     = $session ? floatval($session->get('wa_discount', 0))   : 0;

  // Subtotais por tipo
  list($service_subtotal, $non_service_subtotal) = wa_fx_split_subtotals($cart);

  // Se gratuito => subtrai tudo que NÃO é serviço (passagens isentas)
  if ( $gratis && $non_service_subtotal > 0 ) {
    $cart->add_fee('Isenção das passagens (gratuito)', -1 * $non_service_subtotal, false);
  }

  // Base do desconto:
  // - Se gratuito: aplica sobre o SERVIÇO (apenas).
  // - Se normal: aplica sobre o subtotal de itens.
  $base = $gratis ? $service_subtotal : $cart->get_cart_contents_total();

  // Base do desconto corrigida:
// - Se gratuito: aplica sobre o SERVIÇO (apenas).
// - Se normal: aplica sobre subtotal + taxas já aplicadas
if ( $gratis ) {
    $base = $service_subtotal;
} else {
    // Subtotal dos produtos
    $base = $cart->get_cart_contents_total();

    // Somar fees já existentes, se houver
    $fees_total = 0.0;
    if ( method_exists( $cart, 'get_fees' ) ) {
        foreach ( (array) $cart->get_fees() as $fee ) {
            $fees_total += isset($fee->amount) ? floatval($fee->amount) : 0.0;
        }
    }
    $base += $fees_total;
}



  if ($val > 0 && $base > 0) {
    if ($type === 'percent') {
      $val = min($val, 100);
      $amount = $base * ($val / 100.0);
      $label  = sprintf('Desconto do Operador (%s%%)', wc_format_localized_decimal($val));
    } else {
      $amount = $val;
      $label  = 'Desconto do Operador';
    }
    $amount = min($amount, $base);
    if ($amount > 0) {
      $cart->add_fee($label, -1 * $amount, false); // não tributável
    }
  }
}, 20);

/* -------------------------------------------------
 * Checkout “sem atrito” (apenas no CLÁSSICO)
 * -------------------------------------------------*/
add_filter('woocommerce_checkout_fields', function($fields){
  if (defined('REST_REQUEST') && REST_REQUEST) return $fields; // Blocks: não alterar
  $method = isset($_POST['payment_method']) ? sanitize_text_field($_POST['payment_method']) : '';
  if (!$method && isset($_POST['post_data'])) { $out=[]; parse_str(wp_unslash($_POST['post_data']), $out); $method = isset($out['payment_method']) ? sanitize_text_field($out['payment_method']) : ''; }
  if (strpos($method, 'avulso_') === 0) {
    foreach (['billing','shipping'] as $sec) {
      if (!empty($fields[$sec])) { foreach ($fields[$sec] as &$f) { $f['required'] = false; } }
    }
    if (isset($fields['account'])) { foreach ($fields['account'] as &$f) { $f['required'] = false; } }
  }
  return $fields;
}, 9999);

/* Clássico: remove erros se método avulso; Blocks: não altera */
add_action('woocommerce_after_checkout_validation', function($posted, $errors){
  if (defined('REST_REQUEST') && REST_REQUEST) { return; } // Blocks
  $method = isset($posted['payment_method']) ? sanitize_text_field($posted['payment_method']) : '';
  if (strpos($method, 'avulso_') === 0 && is_wp_error($errors)) {
    foreach ($errors->get_error_codes() as $code) { $errors->remove($code); }
    wc_clear_notices();
  }
}, 5, 2);

/* Preencher mínimos no pedido (nome/email) se faltarem */
add_action('woocommerce_checkout_create_order', function($order, $data){
  if ( ! $order instanceof WC_Order ) return;
  if ( ! $order->get_billing_first_name() ) $order->set_billing_first_name('Cliente');
  if ( ! $order->get_billing_last_name() )  $order->set_billing_last_name('Local');
  if ( ! $order->get_billing_email() )      $order->set_billing_email('cliente+'.time().'@example.com');
}, 5, 2);

/* -------------------------------------------------
 * UI do Desconto — Checkout CLÁSSICO
 * -------------------------------------------------*/
add_action('woocommerce_review_order_before_payment', function(){
  if ( ! function_exists('WC') || ! WC()->session ) return;
  $type   = WC()->session->get('wa_discount_type', 'fixed');
  $val    = 0; // inicia zerado
  $gratis = (bool) WC()->session->get('wa_mode_gratis', false);
  ?>
  <div id="wa-operator-discount-panel" style="margin-bottom:16px;border:1px solid #e5e7eb;border-radius:8px;padding:12px;">
    <p style="margin:0 0 8px;"><strong>Desconto do Operador</strong></p>
    <div style="display:flex;gap:8px;align-items:center;flex-wrap:wrap;">
      <label for="wa_discount" style="margin:0;">Valor:</label>
      <input type="text" class="input-text" id="wa_discount" name="wa_discount" inputmode="decimal" autocomplete="off"
             style="max-width:160px" placeholder="ex: 10,00"
             value="<?php echo esc_attr( wc_format_localized_decimal( floatval($val) ) ); ?>">
      <select id="wa_discount_type" name="wa_discount_type" class="select" style="max-width:180px;">
        <option value="fixed"   <?php selected($type,'fixed'); ?>>Fixo (R$)</option>
        <option value="percent" <?php selected($type,'percent'); ?>>% Percentual</option>
      </select>

      <label style="display:flex;align-items:center;gap:6px;margin-left:12px;">
        <input type="checkbox" id="wa_discount_gratis" <?php checked($gratis, true); ?>>
        <span>Somente cobrar serviço (gratuito)</span>
      </label>
    </div>
    <small>Se “gratuito” ativo, o total exibido será apenas o valor do serviço (passagens isentas). O desconto aplica sobre o serviço.</small>
  </div>

  <script>
  (function($){
    if (window.__waDiscClassicBound) return; window.__waDiscClassicBound = true;

    function formatBRL(n){ try { return new Intl.NumberFormat('pt-BR',{style:'currency',currency:'BRL'}).format(n); } catch(e){ n=(n||0); return 'R$ '+n.toFixed(2).replace('.',','); } }
    function toNumberBR(str){ str=(str||'').toString(); str=str.replace(/[^\d,.-]/g,''); if (str.indexOf(',')!==-1){ str=str.replace(/\./g,'').replace(',', '.'); } var n=parseFloat(str); return isNaN(n)||n<0?0:n; }
    function maskBRLInput(el){ let only=(el.value||'').replace(/\D/g,''); if (!only) only='0'; let n=parseInt(only,10)/100; el.value=formatBRL(n); }

    let placing=false, lastSnap='', sending=false, reSubmitting=false;

    const snap = ()=> JSON.stringify({
      t:$('#wa_discount_type').val(),
      v:$('#wa_discount').val(),
      g:$('#wa_discount_gratis').is(':checked')?1:0
    });

    function blockCheckout(msg){
      try { $('form.checkout').block({ message: msg || 'Aplicando...', overlayCSS:{ background:'#fff', opacity:0.6 } }); } catch(e){}
    }
    function unblockCheckout(){ try { $('form.checkout').unblock(); } catch(e){} }

    function sendAjax(callback){
      var type = $('#wa_discount_type').val();
      var val  = toNumberBR($('#wa_discount').val());
      var gratis = $('#wa_discount_gratis').is(':checked') ? 1 : 0;
      sending = true;
      blockCheckout('Aplicando...');
      $.post('<?php echo admin_url('admin-ajax.php'); ?>', { action:'wa_set_discount', type:type, value:val, gratis:gratis }, function(){
        $(document.body).trigger('update_checkout');
      }).always(function(){
        sending = false;
        unblockCheckout();
        if (typeof callback === 'function') setTimeout(callback, 120);
      });
    }

    function applyFormatting(){
      var type=$('#wa_discount_type').val(), $inp=$('#wa_discount'), num=toNumberBR($inp.val());
      if (type==='fixed'){ $inp.val( formatBRL(num) ); } else { $inp.val( String(num.toFixed(2)).replace('.',',') ); }
    }

    $(document).on('input','#wa_discount',function(){ if ($('#wa_discount_type').val()==='fixed'){ maskBRLInput(this); } });

    $(document).on('blur','#wa_discount',function(){
      applyFormatting();
      const s = snap(); if (placing || s===lastSnap) return; lastSnap=s; sendAjax();
    });

    $(document).on('change','#wa_discount_type, #wa_discount_gratis',function(){
      applyFormatting();
      const s = snap(); if (placing || s===lastSnap) return; lastSnap=s; sendAjax();
    });

    $(document.body).on('checkout_place_order', function(e){
      const s = snap();

      if (sending) { e.preventDefault(); return false; }

      if (s !== lastSnap) {
        e.preventDefault();
        lastSnap = s;
        sendAjax(function(){
          if (!reSubmitting){
            reSubmitting = true;
            $('form.checkout').submit();
          } else {
            reSubmitting = false;
          }
        });
        return false;
      }

      placing = true;
      return true;
    });

    $(document).on('updated_checkout checkout_error', function(){
      placing = false; reSubmitting = false;
      lastSnap = snap();
      unblockCheckout();
    });

    function init(){ applyFormatting(); lastSnap = snap(); }
    $(document).ready(init); $(document.body).on('updated_checkout', init);
  })(jQuery);
  </script>
  <?php
});

/* -------------------------------------------------
 * Gateways Avulsos (Clássico)
 * -------------------------------------------------*/
add_filter('woocommerce_payment_gateways', function($gws){
  $gws[] = 'WC_Gateway_Avulso_Dinheiro_FX';
  $gws[] = 'WC_Gateway_Avulso_Pix_FX';
  $gws[] = 'WC_Gateway_Avulso_Debito_FX';
  $gws[] = 'WC_Gateway_Avulso_Credito_FX';
  return $gws;
});

add_action('plugins_loaded', function(){
  if ( ! class_exists('WC_Payment_Gateway') ) return;

  abstract class WC_Gateway_Avulso_Base_FX extends WC_Payment_Gateway {
    public function __construct($id, $title, $description) {
      $this->id = $id;
      $this->method_title       = $title;
      $this->method_description = $description;
      $this->title              = $title;
      $this->has_fields         = false;
      $this->supports           = ['products'];

      $this->init_form_fields();
      $this->init_settings();
      $this->enabled = $this->get_option('enabled', 'yes');

      add_action('woocommerce_update_options_payment_gateways_' . $this->id, [$this, 'process_admin_options']);
    }
    public function init_form_fields() {
      $this->form_fields = [
        'enabled' => [
          'title'=>'Ativar/Desativar','type'=>'checkbox','label'=>'Ativar este método de pagamento','default'=>'yes'
        ],
      ];
    }
    public function is_available() {
      return 'yes' === $this->enabled && ( WC()->cart ? WC()->cart->needs_payment() : true );
    }
    public function validate_fields(){ return true; }
    public function process_payment($order_id) {
      $order = wc_get_order($order_id);
      if ($order) {
        $order->calculate_totals();
        $order->payment_complete();
        $order->update_status('completed', sprintf('Pagamento confirmado via: %s', $this->method_title));
        wc_reduce_stock_levels($order_id);
        if ( function_exists('WC') && WC()->cart ) { WC()->cart->empty_cart(); }
      }
      return ['result'=>'success','redirect'=>$this->get_return_url($order)];
    }
  }

  class WC_Gateway_Avulso_Dinheiro_FX extends WC_Gateway_Avulso_Base_FX {
    public function __construct(){ parent::__construct('avulso_dinheiro_fx','Avulso - Dinheiro','Pagamento feito em dinheiro.'); }
  }
  class WC_Gateway_Avulso_Pix_FX extends WC_Gateway_Avulso_Base_FX {
    public function __construct(){ parent::__construct('avulso_pix_fx','Avulso - Pix','Pagamento via Pix.'); }
  }
  class WC_Gateway_Avulso_Debito_FX extends WC_Gateway_Avulso_Base_FX {
    public function __construct(){ parent::__construct('avulso_debito_fx','Avulso - Débito','Pagamento via cartão de débito.'); }
  }
  class WC_Gateway_Avulso_Credito_FX extends WC_Gateway_Avulso_Base_FX {
    public function __construct(){ parent::__construct('avulso_credito_fx','Avulso - Crédito','Pagamento via cartão de crédito.'); }
  }
});

/* -------------------------------------------------
 * SUPORTE AO CHECKOUT BLOCKS (registro front-end + UI do desconto)
 * -------------------------------------------------*/
add_action('woocommerce_blocks_loaded', function(){
  if ( ! class_exists('\Automattic\WooCommerce\Blocks\Payments\Integrations\AbstractPaymentMethodType') ) return;

  class WA_Blocks_Gateway_FX extends \Automattic\WooCommerce\Blocks\Payments\Integrations\AbstractPaymentMethodType {
    protected $name;
    protected $settings = [];

    public function __construct( $gateway_id ) { $this->name = $gateway_id; }

    public function initialize() {
      $this->settings = get_option('woocommerce_' . $this->name . '_settings', []);
    }

    public function is_active() {
      $enabled = isset($this->settings['enabled']) ? $this->settings['enabled'] : 'yes';
      return 'yes' === $enabled;
    }

    public function get_payment_method_data() {
      $defaults = [
        'avulso_dinheiro_fx' => ['title'=>'Avulso - Dinheiro', 'desc'=>'Pagamento feito em dinheiro.'],
        'avulso_pix_fx'      => ['title'=>'Avulso - Pix',      'desc'=>'Pagamento via Pix.'],
        'avulso_debito_fx'   => ['title'=>'Avulso - Débito',   'desc'=>'Pagamento via cartão de débito.'],
        'avulso_credito_fx'  => ['title'=>'Avulso - Crédito',  'desc'=>'Pagamento via cartão de crédito.'],
      ];
      $d = $defaults[$this->name] ?? ['title'=>'Avulso','desc'=>''];
      $title = $this->settings['title'] ?? $d['title'];
      $desc  = $this->settings['description'] ?? $d['desc'];
      return ['title'=>$title, 'description'=>$desc, 'supports'=>['products']];
    }

    public function get_payment_method_script_handles() {
      $handle = 'wa-avulso-blocks-' . $this->name;
      $deps   = [ 'wc-blocks-registry', 'wc-settings', 'wp-element', 'wp-i18n', 'wp-data' ];
      wp_register_script( $handle, '', $deps, 'fx3.2', true );

      wp_localize_script($handle, 'WA_AVULSO_FX', [
        'ajax_url' => admin_url('admin-ajax.php'),
      ]);

      $safe = esc_js( $this->name );
      $js = "(function(){
  if (window.__WAAvulsoFX__{$safe}) return; window.__WAAvulsoFX__{$safe} = true;
  var reg = window.wc && (window.wc.wcBlocksRegistry || window.wc.blocksRegistry);
  var settings = window.wc && (window.wc.wcSettings || window.wc.settings);
  if (!reg || !reg.registerPaymentMethod || !settings || !settings.getPaymentMethodData) return;

  var el = window.wp && wp.element ? wp.element : null;
  var i18n = window.wp && wp.i18n ? wp.i18n : null;
  var data = settings.getPaymentMethodData('{$safe}') || {};
  var __ = i18n ? i18n.__ : function(s){return s;};

  function toNumberBR(str){
    str = (str||'').toString();
    str = str.replace(/[^\\d,.-]/g,'');
    if (str.indexOf(',') !== -1){ str = str.replace(/\\./g,'').replace(',', '.'); }
    var n = parseFloat(str);
    return isNaN(n) || n<0 ? 0 : n;
  }
  function formatBRL(n){
    try { return new Intl.NumberFormat('pt-BR',{style:'currency',currency:'BRL'}).format(n); }
    catch(e){ n=(n||0); return 'R$ '+n.toFixed(2).replace('.',','); }
  }

  var DiscountContent = function(){
    var React = el;
    var [type, setType] = React.useState('fixed');
    var [val, setVal]   = React.useState('0,00'); // inicia zerado
    var [gratis, setGratis] = React.useState(false);
    var [lastSnap, setLastSnap] = React.useState(JSON.stringify({t:'fixed', v:'0,00', g:0}));
    var saving = React.useRef(false);
    var saveQueued = React.useRef(false);
    var debounceTimer = React.useRef(null);

    function invalidateTotals(){
      try {
        if (window.wp && wp.data && wp.data.dispatch) {
          var d = wp.data.dispatch('wc/store/cart');
          if (d && d.invalidateResolution) {
            try{ d.invalidateResolution('getCartTotals'); }catch(e){}
            try{ d.invalidateResolution('getCart'); }catch(e){}
          }
        }
      } catch(e){}
    }

    function saveNow(){
      var snap = JSON.stringify({t:type, v:val, g: (gratis?1:0)});
      if (saving.current){ saveQueued.current = true; return Promise.resolve(); }
      saving.current = true;
      var form = new FormData();
      form.append('action','wa_set_discount');
      form.append('type', type);
      form.append('value', toNumberBR(val));
      form.append('gratis', gratis?1:0);
      return fetch(WA_AVULSO_FX.ajax_url, { method:'POST', body: form })
        .then(function(){ invalidateTotals(); })
        .catch(function(){})
        .finally(function(){
          saving.current = false;
          setLastSnap(snap);
          if (saveQueued.current){ saveQueued.current = false; saveNow(); }
        });
    }

    function saveDebounced(){
      if (debounceTimer.current) clearTimeout(debounceTimer.current);
      debounceTimer.current = setTimeout(function(){ saveNow(); }, 250);
    }

    function onBlur(){ saveNow(); }
    function onInput(e){
      var v = e.target.value || '';
      if (type === 'fixed'){
        var only = v.replace(/\\D/g,''); if (!only) only='0';
        var n = parseInt(only,10)/100;
        v = formatBRL(n);
      }
      setVal(v);
      saveDebounced();
    }
    function onTypeChange(e){ setType(e.target.value); saveDebounced(); }
    function onGratisChange(e){ setGratis(e.target.checked); saveDebounced(); }

    // Intercepta o click no botão de finalizar (Blocks)
    document.addEventListener('click', function(ev){
      try {
        var target = ev.target;
        if (!target) return;
        var isPlaceOrder = target.closest('.wc-block-components-checkout-place-order-button, button[name=\"place_order\"]');
        if (!isPlaceOrder) return;

        var currentSnap = JSON.stringify({t:type, v:val, g:(gratis?1:0)});
        if (currentSnap !== lastSnap || saving.current){
          ev.preventDefault();
          ev.stopPropagation();
          saveNow().then(function(){
            setTimeout(function(){ target.click(); }, 120);
          });
        }
      } catch(e){}
    }, true);

    // Salva quando o método é montado
    React.useEffect(function(){ saveNow(); }, []);

    return el.createElement('div', { className: 'wa-avulso-blocks-box', style:{marginTop:'8px'} },
      el.createElement('div', { style:{display:'flex',gap:'8px',alignItems:'center',flexWrap:'wrap'} },
        el.createElement('label', { htmlFor:'wa_discount_blocks' }, __('Desconto do Operador','wa')),
        el.createElement('input', {
          id:'wa_discount_blocks', type:'text', value: val,
          onChange:onInput, onBlur:onBlur, placeholder: 'ex: 10,00', style:{maxWidth:'160px'}
        }),
        el.createElement('select', {
          value: type, onChange: onTypeChange, onBlur: onBlur, style:{maxWidth:'180px'}
        },
          el.createElement('option', {value:'fixed'},  __('Fixo (R$)','wa')),
          el.createElement('option', {value:'percent'}, __('% Percentual','wa'))
        ),
        el.createElement('label', { style:{display:'flex',alignItems:'center',gap:'6px',marginLeft:'12px'} },
          el.createElement('input', { type:'checkbox', checked: !!gratis, onChange: onGratisChange }),
          el.createElement('span', null, __('Somente cobrar serviço (gratuito)','wa'))
        )
      ),
      el.createElement('small', null, __('Se “gratuito” ativo, o total exibido será apenas o valor do serviço; o desconto aplica sobre o serviço.','wa'))
    );
  };

  reg.registerPaymentMethod({
    name: '{$safe}',
    label: data.title || '{$safe}',
    content: DiscountContent,
    edit: DiscountContent,
    canMakePayment: function(){ return true; },
    ariaLabel: data.title || '{$safe}',
    supports: { features: ['products'] }
  });
})();";
      wp_add_inline_script( $handle, $js );
      return [ $handle ];
    }
  }

  add_action('woocommerce_blocks_payment_method_type_registration', function( $registry ){
    foreach ( ['avulso_dinheiro_fx','avulso_pix_fx','avulso_debito_fx','avulso_credito_fx'] as $id ) {
      $registry->register( new WA_Blocks_Gateway_FX( $id ) );
    }
  });
});

/* -------------------------------------------------
 * HOTFIX Store API (Blocks)
 * -------------------------------------------------*/
add_action('woocommerce_store_api_checkout_order_processed', function($order){
  if ( ! $order instanceof WC_Order ) return;
  $pm = $order->get_payment_method(); // ex.: avulso_dinheiro_fx
  if (strpos($pm, 'avulso_') === 0) {
    $order->payment_complete();
    $order->update_status('completed', 'Pagamento confirmado via Avulso (Store API).');
    wc_reduce_stock_levels( $order->get_id() );
  }
}, 20);
