<?php
/**
 * Template Name: Página Bem-vindo Clean
 */
if ( ! defined('ABSPATH') ) exit;

get_header();

$current_user = wp_get_current_user();
$display_name = $current_user->display_name ?: $current_user->user_login;
$roles        = wp_roles()->get_names();
$user_roles   = array();
foreach ( (array) $current_user->roles as $r ) {
    $user_roles[] = isset($roles[$r]) ? $roles[$r] : ucfirst($r);
}
?>


<?php
/**
 * Template Name: Baixa Rápida de Fretes (Teclado Turbo)
 * Description: Últimos 365 dias, somente EM ABERTO. DataTables + Select2 + Chart.js. Atalhos de teclado e fluxo 100% focado no operador.
 */
if (!defined('ABSPATH')) exit;
date_default_timezone_set('America/Manaus');

/* ==================== Helpers ==================== */
function brq_br2f($v){
  if ($v === '' || $v === null) return 0.0;
  if (is_numeric($v)) return (float)$v;
  $v = str_replace(['.', ' '], '', (string)$v);
  $v = str_replace(',', '.', $v);
  return (float)(is_numeric($v)? $v : 0);
}
function brq_f2br($f){ return number_format((float)$f, 2, ',', '.'); }

/* ==================== POST: aplicar baixa ==================== */
$notice=''; $error='';
if ($_SERVER['REQUEST_METHOD']==='POST' && isset($_POST['baixa_submit'])) {
  if (!current_user_can('edit_posts')) {
    $error = 'Sem permissão.';
  } elseif (!isset($_POST['baixa_nonce']) || !wp_verify_nonce($_POST['baixa_nonce'], 'baixa_dt')) {
    $error = 'Falha de segurança (nonce).';
  } else {
    $ids = array_filter(array_map('intval', explode(',', sanitize_text_field($_POST['sel_ids'] ?? ''))));
    $marcar_entregue = !empty($_POST['entregar']);

    $din = brq_br2f($_POST['dinheiro'] ?? '0');
    $pix = brq_br2f($_POST['pix'] ?? '0');
    $deb = brq_br2f($_POST['debito'] ?? '0');
    $cre = brq_br2f($_POST['credito'] ?? '0');
    $des = brq_br2f($_POST['desconto'] ?? '0');

    if (!$ids) {
      $error = 'Selecione ao menos um frete.';
    } elseif (($din+$pix+$deb+$cre+$des) <= 0) {
      $error = 'Informe algum valor de pagamento ou desconto.';
    } else {
      $posts = get_posts([
        'post_type'      => 'encomendas',
        'post__in'       => $ids,
        'posts_per_page' => -1,
        'orderby'        => 'date',
        'order'          => 'ASC',
        'post_status'    => ['publish','pending','draft','private'],
        'meta_query'     => [
          ['key'=>'tipocat','value'=>'Frete','compare'=>'=']
        ]
      ]);
      $ok=0; $agora=current_time('Y-m-d H:i');
      $current_user   = wp_get_current_user();
      $operador_id    = (int) $current_user->ID;
      $operador_nome  = $current_user->display_name;

      foreach ($posts as $p){
        $pid = $p->ID;

        $tot  = brq_br2f(get_post_meta($pid,'valor_total_produtos',true));
        $pago = brq_br2f(get_post_meta($pid,'valor_total_pago',true));
        $dins = brq_br2f(get_post_meta($pid,'valor_dinheiro',true));
        $pixs = brq_br2f(get_post_meta($pid,'valor_pix',true));
        $debs = brq_br2f(get_post_meta($pid,'valor_debito',true));
        $cres = brq_br2f(get_post_meta($pid,'valor_credito',true));
        $dess = brq_br2f(get_post_meta($pid,'valor_desconto',true));

        $devido = max(0, $tot - $pago - $dess);
        if ($devido <= 0.001) continue;

        // Prioridade: Desconto → Dinheiro → Pix → Débito → Crédito
        if ($des>0 && $devido>0){ $use=min($des,$devido); $dess+=$use; $des-=$use; $devido-=$use; }
        if ($din>0 && $devido>0){ $use=min($din,$devido); $dins+=$use; $pago+=$use; $din-=$use; $devido-=$use; }
        if ($pix>0 && $devido>0){ $use=min($pix,$devido); $pixs+=$use; $pago+=$use; $pix-=$use; $devido-=$use; }
        if ($deb>0 && $devido>0){ $use=min($deb,$devido); $debs+=$use; $pago+=$use; $deb-=$use; $devido-=$use; }
        if ($cre>0 && $devido>0){ $use=min($cre,$devido); $cres+=$use; $pago+=$use; $cre-=$use; $devido-=$use; }

        update_post_meta($pid,'valor_desconto',     brq_f2br($dess));
        update_post_meta($pid,'valor_dinheiro',     brq_f2br($dins));
        update_post_meta($pid,'valor_pix',          brq_f2br($pixs));
        update_post_meta($pid,'valor_debito',       brq_f2br($debs));
        update_post_meta($pid,'valor_credito',      brq_f2br($cres));
        update_post_meta($pid,'valor_total_pago',   brq_f2br($pago));

        $novo_devido = max(0, $tot - $pago - $dess);
        update_post_meta($pid,'valor_total_devido', brq_f2br($novo_devido));
        update_post_meta($pid,'data_pagamento',     $agora);
        update_post_meta($pid, 'operador_lancamento', $operador_id);

        if ($marcar_entregue && $novo_devido <= 0.001) update_post_meta($pid,'status','Entregue');

        $ok++;
        if (($din+$pix+$deb+$cre+$des) <= 0.001) break;
      }
      $notice = $ok.' frete(s) atualizados.';
    }
  }
}

/* ==================== Consulta inicial: EM ABERTO + últimos 365 dias ==================== */
global $wpdb;
$days_default = 365;
$after = wp_date('Y-m-d', current_time('timestamp') - ($days_default * DAY_IN_SECONDS)) . ' 00:00:00';

$sql = "
SELECT
  p.ID,
  DATE(p.post_date) AS emissao,
  rem.meta_value  AS rem,
  dstn.meta_value AS dest,
  org.meta_value  AS origem,
  dsto.meta_value AS destino,
  CAST(REPLACE(REPLACE(COALESCE(tot.meta_value,'0'), '.', ''), ',', '.') AS DECIMAL(15,2)) AS faturado,
  CAST(REPLACE(REPLACE(COALESCE(pag.meta_value,'0'), '.', ''), ',', '.') AS DECIMAL(15,2)) AS pago,
  CASE
    WHEN dev.meta_value IS NOT NULL AND dev.meta_value <> ''
      THEN CAST(REPLACE(REPLACE(dev.meta_value, '.', ''), ',', '.') AS DECIMAL(15,2))
    ELSE GREATEST(
      0,
      CAST(REPLACE(REPLACE(COALESCE(tot.meta_value,'0'), '.', ''), ',', '.') AS DECIMAL(15,2))
      - CAST(REPLACE(REPLACE(COALESCE(pag.meta_value,'0'), '.', ''), ',', '.') AS DECIMAL(15,2))
      - CAST(REPLACE(REPLACE(COALESCE(des.meta_value,'0'), '.', ''), ',', '.') AS DECIMAL(15,2))
    )
  END AS devido,
  st.meta_value   AS status
FROM {$wpdb->posts} p
JOIN {$wpdb->postmeta} tip
  ON tip.post_id = p.ID AND tip.meta_key='tipocat' AND tip.meta_value='Frete'
LEFT JOIN {$wpdb->postmeta} rem
  ON rem.post_id = p.ID AND rem.meta_key='nome_remetente'
LEFT JOIN {$wpdb->postmeta} dstn
  ON dstn.post_id = p.ID AND dstn.meta_key='nome_destinatario'
LEFT JOIN {$wpdb->postmeta} org
  ON org.post_id = p.ID AND org.meta_key='origem'
LEFT JOIN {$wpdb->postmeta} dsto
  ON dsto.post_id = p.ID AND dsto.meta_key='destino'
LEFT JOIN {$wpdb->postmeta} tot
  ON tot.post_id = p.ID AND tot.meta_key='valor_total_produtos'
LEFT JOIN {$wpdb->postmeta} pag
  ON pag.post_id = p.ID AND pag.meta_key='valor_total_pago'
LEFT JOIN {$wpdb->postmeta} des
  ON des.post_id = p.ID AND des.meta_key='valor_desconto'
LEFT JOIN {$wpdb->postmeta} dev
  ON dev.post_id = p.ID AND dev.meta_key='valor_total_devido'
LEFT JOIN {$wpdb->postmeta} st
  ON st.post_id = p.ID AND st.meta_key='status'
WHERE p.post_type='encomendas'
  AND p.post_status IN ('publish','pending','draft','private')
  AND p.post_date >= %s
HAVING devido > 0.001
ORDER BY p.post_date ASC;
";
$rows_db = $wpdb->get_results($wpdb->prepare($sql, $after), ARRAY_A);

/* ==================== Render ==================== */

if (!current_user_can('edit_posts')) {
  echo '<div class="wrap"><h2>Sem permissão</h2><p>Faça login com um usuário que possa editar encomendas.</p></div>';
  get_footer(); exit;
}
?>
 <link href="https://cdn.datatables.net/1.13.8/css/dataTables.bootstrap5.min.css" rel="stylesheet">
<link href="https://cdn.datatables.net/buttons/2.4.2/css/buttons.bootstrap5.min.css" rel="stylesheet">
<link href="https://cdn.datatables.net/responsive/2.5.0/css/responsive.bootstrap5.min.css" rel="stylesheet">
<link href="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/css/select2.min.css" rel="stylesheet">
<link href="https://cdn.jsdelivr.net/npm/select2-bootstrap-5-theme@1.3.0/dist/select2-bootstrap-5-theme.min.css" rel="stylesheet">
<link rel="stylesheet" href="https://cdn.jsdelivr.net/npm/flatpickr/dist/flatpickr.min.css">
<style>
  .container-xxl{max-width:1400px}
  .money{font-variant-numeric:tabular-nums}
  .sticky-top-lite{position:sticky;top:0;z-index:0;background:#fff;border-bottom:1px solid #eee}
  .totals .badge{font-size:0.95rem}
  .card-lite{border:1px solid #e5e7eb;border-radius:12px}
  #offGrafico.offcanvas-end{ width: 420px; } @media (max-width: 576px){ #offGrafico.offcanvas-end{ width: 100%; } }
  /* Spotlight */
  #kb-spotlight { position: fixed; inset: 0; display: none; z-index: 2000;
    background: rgba(15,23,42,.35); backdrop-filter: blur(3px); }
  #kb-spotlight .box { width: min(720px, 92vw); margin: 12vh auto 0; background: #fff;
    border-radius: 14px; box-shadow: 0 20px 60px rgba(0,0,0,.25); border: 1px solid #e5e7eb; padding: 14px; }
  #kb-spotlight .scope { font-size: 12px; color:#475569; margin-bottom:6px }
  #kb-spotlight input { width: 100%; padding: 14px 16px; font-size: 18px; border: 1px solid #cbd5e1; border-radius: 10px; }
  #kb-spotlight .kb-list{ margin:10px 0 0; padding:0; list-style:none; max-height:300px; overflow:auto; border:1px solid #e5e7eb; border-radius:10px; }
  #kb-spotlight .kb-list li{ padding:10px 12px; cursor:pointer; border-bottom:1px solid #eef2f7; white-space:nowrap; overflow:hidden; text-overflow:ellipsis; }
  #kb-spotlight .kb-list li:last-child{ border-bottom:none; }
  #kb-spotlight .kb-list li.active{ background:#f0f9ff; }
</style>

<div class="container-xxl py-3">
  <div class="sticky-top-lite py-2 d-flex align-items-center justify-content-between">
    <h4 class="mb-0"><span class="text-danger">Em aberto</span> últimos 365 dias</h4>
    <div class="d-flex align-items-center gap-2">
      <button class="btn btn-outline-primary" data-bs-toggle="offcanvas" data-bs-target="#offGrafico">📊 Gráfico</button>
      <div class="d-flex align-items-center gap-3 totals ms-2">
        <span class="badge bg-danger">Devedor (visíveis): <span id="tv-dev">R$ 0,00</span></span>
        <span class="badge bg-success">Selecionado • Devedor: <span id="ts-dev">R$ 0,00</span></span>
        <button class="btn btn-success" data-bs-toggle="modal" data-bs-target="#modalBaixa" id="btnAbrirBaixa" disabled>Aplicar baixa</button>
      </div>
    </div>
  </div>

  <div class="alert alert-info mt-3 card-lite">
    <b>Atalhos (100% teclado)</b> —
    <span class="me-2">F3 Destinatário (spotlight)</span>
    <span class="me-2">F4 Abrir pagamento (modal)</span>
    <span class="me-2">F5 Origem (spotlight)</span>
    <span class="me-2">F6 Destino (spotlight)</span>
    <span class="me-2">F7 Selecionar visíveis</span>
    <span class="me-2">F8 Abrir modal (sem preencher)</span>
    <span class="me-2">F9 Limpar filtros</span>
    <span>ESC (no Spotlight) Aplica e fecha</span><br>
    <small>ENTER na busca/Select2/Spotlight ⇒ filtra, seleciona visíveis e abre o modal (Dinheiro auto-preenchido).</small>
  </div>

  <!-- Offcanvas do Gráfico -->
  <div class="offcanvas offcanvas-end" tabindex="-1" id="offGrafico">
    <div class="offcanvas-header">
      <h5 class="offcanvas-title">Top Devedores</h5>
      <button type="button" class="btn-close" data-bs-dismiss="offcanvas"></button>
    </div>
    <div class="offcanvas-body">
      <div class="row g-2 mb-3">
        <div class="col-12">
          <label class="form-label mb-1">Agrupar por</label>
          <select id="chart-group" class="form-select">
            <option value="4">Destinatário</option>
            <option value="3">Remetente</option>
            <option value="6">Destino</option>
            <option value="5">Origem</option>
          </select>
        </div>
        <div class="col-6">
          <label class="form-label mb-1">Tipo</label>
          <select id="chart-type" class="form-select">
            <option value="doughnut">Pizza (doughnut)</option>
            <option value="bar">Barras</option>
          </select>
        </div>
        <div class="col-6">
          <label class="form-label mb-1">Top N</label>
          <input type="number" id="chart-topn" class="form-control" value="10" min="3" max="30">
        </div>
      </div>
      <div class="card-lite p-2" style="height:260px">
        <canvas id="debtChart"></canvas>
      </div>
      <small class="text-muted d-block mt-2">O gráfico usa as linhas <b>visíveis</b> na tabela.</small>
    </div>
  </div>

  <!-- Filtro por período -->
  <div class="row g-2 align-items-end my-2">
    <div class="col-md-4">
      <label class="form-label mb-1">Período</label>
      <input type="text" id="f-range" class="form-control" placeholder="Selecione o período">
    </div>
    <div class="col-md-8 d-flex flex-wrap gap-2">
      <button class="btn btn-outline-secondary" id="r-7"   type="button">Últimos 7 dias</button>
      <button class="btn btn-outline-secondary" id="r-30"  type="button">Últimos 30 dias</button>
      <button class="btn btn-outline-secondary" id="r-365" type="button">Últimos 365 dias</button>
      <button class="btn btn-outline-danger"    id="r-clear" type="button">Limpar período</button>
    </div>
  </div>

  <!-- Filtros principais -->
  <div class="row g-2 my-2">
    <div class="col-md-3">
      <label class="form-label mb-1">Remetente</label>
      <select id="f-rem" class="form-select select2" multiple></select>
    </div>
    <div class="col-md-3">
      <label class="form-label mb-1">Destinatário</label>
      <select id="f-dest" class="form-select select2" multiple></select>
    </div>
    <div class="col-md-3">
      <label class="form-label mb-1">Origem</label>
      <select id="f-org" class="form-select select2" multiple></select>
    </div>
    <div class="col-md-3">
      <label class="form-label mb-1">Destino</label>
      <select id="f-dsti" class="form-select select2" multiple></select>
    </div>
    <div class="col-12 d-flex justify-content-end">
      <button class="btn btn-outline-secondary mt-2" id="btnClearFilters">Limpar filtros</button>
    </div>
  </div>

  <!-- Tabela -->
  <div class="table-responsive">
    <table id="tFretes" class="table table-striped table-bordered align-middle nowrap" style="width:100%">
      <thead>
        <tr>
          <th style="width:34px;"><input type="checkbox" id="chkAll"></th>
          <th>Nº</th>
          <th>Emissão</th>
          <th>Remetente</th>
          <th>Destinatário</th>
          <th>Origem</th>
          <th>Destino</th>
          <th>Faturado</th>
          <th>Pago</th>
          <th>Devedor</th>
          <th>Status</th>
        </tr>
      </thead>
      <tbody>
      <?php if ($rows_db): foreach($rows_db as $r):
        $id = (int)$r['ID'];
        $emissao = esc_html(date_i18n('d/m/Y', strtotime($r['emissao'])));
        $emissao_iso = esc_attr($r['emissao']);
        $rem = esc_html($r['rem']);
        $dest = esc_html($r['dest']);
        $org = esc_html($r['origem']);
        $dst = esc_html($r['destino']);
        $fat = (float)$r['faturado'];
        $pag = (float)$r['pago'];
        $dev = (float)$r['devido'];
        $st  = esc_html($r['status']);
      ?>
        <tr data-id="<?php echo $id; ?>" data-date="<?php echo $emissao_iso; ?>">
          <td class="text-center"><input type="checkbox" class="rowchk"></td>
          <td><?php echo $id; ?></td>
          <td data-sort="<?php echo $emissao_iso; ?>"><?php echo $emissao; ?></td>
          <td><?php echo $rem; ?></td>
          <td><?php echo $dest; ?></td>
          <td><?php echo $org; ?></td>
          <td><?php echo $dst; ?></td>
          <td class="text-end money">R$ <?php echo brq_f2br($fat); ?></td>
          <td class="text-end money">R$ <?php echo brq_f2br($pag); ?></td>
          <td class="text-end money dev">R$ <?php echo brq_f2br($dev); ?></td>
          <td><?php echo $st; ?></td>
        </tr>
      <?php endforeach; endif; ?>
      </tbody>
      <tfoot>
        <tr>
          <th></th><th colspan="6" class="text-end">Totais (visíveis):</th>
          <th class="text-end">R$ 0,00</th>
          <th class="text-end">R$ 0,00</th>
          <th class="text-end">R$ 0,00</th>
          <th></th>
        </tr>
      </tfoot>
    </table>
  </div>

  <!-- Modal Baixa -->
  <div class="modal fade" id="modalBaixa" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog modal-lg">
      <form method="post" class="modal-content">
        <div class="modal-header">
          <h5 class="modal-title">Aplicar baixa em lote</h5>
          <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
        </div>
        <div class="modal-body">
          <?php wp_nonce_field('baixa_dt','baixa_nonce'); ?>
          <input type="hidden" name="sel_ids" id="sel_ids">
          <div class="row g-3">
            <div class="col-md-12"><label class="form-label">Desconto</label><input type="text" class="form-control money-in" name="desconto" id="in_des" placeholder="0,00"></div>
            <div class="col-md-3"><label class="form-label">Dinheiro</label><input type="text" class="form-control money-in" name="dinheiro" id="in_din" placeholder="0,00"></div>
            <div class="col-md-3"><label class="form-label">Pix</label><input type="text" class="form-control money-in" name="pix" id="in_pix" placeholder="0,00"></div>
            <div class="col-md-3"><label class="form-label">Débito</label><input type="text" class="form-control money-in" name="debito" id="in_deb" placeholder="0,00"></div>
            <div class="col-md-3"><label class="form-label">Crédito</label><input type="text" class="form-control money-in" name="credito" id="in_cre" placeholder="0,00"></div>
          </div>
          <div class="d-flex justify-content-between align-items-center mt-3">
            <div>Selecionado (Devedor): <strong id="sel_dev_modal">R$ 0,00</strong></div>
            <div>Informado: <strong id="inf_modal">R$ 0,00</strong></div>
            <div>Faltando: <strong id="falt_modal" class="text-danger">R$ 0,00</strong></div>
          </div>
          <div class="form-check form-switch mt-2">
            <input class="form-check-input" type="checkbox" id="in_entregar" name="entregar" checked>
            <label class="form-check-label" for="in_entregar">Marcar como <b>Entregue</b> quando quitar</label>
          </div>
          <div class="mt-2 d-flex gap-2">
            <button class="btn btn-outline-secondary" type="button" id="btnQuitarDin">Quitar tudo em Dinheiro</button>
            <button class="btn btn-outline-secondary" type="button" id="btnQuitarPix">Quitar tudo em Pix</button>
            <button class="btn btn-outline-secondary" type="button" id="btnZerar">Zerar</button>
          </div>
        </div>
        <div class="modal-footer">
          <button class="btn btn-secondary" data-bs-dismiss="modal" type="button">Cancelar</button>
          <button class="btn btn-primary" name="baixa_submit" value="1" id="btnConfirm" disabled>Confirmar baixa</button>
        </div>
      </form>
    </div>
  </div>

  <?php if ($notice): ?><div class="alert alert-success mt-3"><?php echo esc_html($notice); ?></div><?php endif; ?>
  <?php if ($error):  ?><div class="alert alert-danger mt-3"><?php echo esc_html($error);  ?></div><?php endif; ?>
</div>

<!-- Spotlight -->
<div id="kb-spotlight" aria-hidden="true">
  <div class="box">
    <div class="scope">Filtrar por <b id="kb-scope-name">Destinatário</b> — digite, use ↑/↓ e <b>Enter</b></div>
    <input type="text" id="kb-input" placeholder="Digite aqui...">
    <ul id="kb-list" class="kb-list"></ul>
    <small>ESC fecha aplicando • Enter adiciona • F4 aplica e abre o modal</small>
  </div>
</div>

<script src="https://code.jquery.com/jquery-3.6.4.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.3/dist/js/bootstrap.bundle.min.js"></script>
<script src="https://cdn.datatables.net/1.13.8/js/jquery.dataTables.min.js"></script>
<script src="https://cdn.datatables.net/1.13.8/js/dataTables.bootstrap5.min.js"></script>
<script src="https://cdn.datatables.net/responsive/2.5.0/js/dataTables.responsive.min.js"></script>
<script src="https://cdn.datatables.net/buttons/2.4.2/js/dataTables.buttons.min.js"></script>
<script src="https://cdn.datatables.net/buttons/2.4.2/js/buttons.bootstrap5.min.js"></script>
<script src="https://cdn.datatables.net/buttons/2.4.2/js/buttons.html5.min.js"></script>
<script src="https://cdn.datatables.net/buttons/2.4.2/js/buttons.print.min.js"></script>
<script src="https://cdnjs.cloudflare.com/ajax/libs/jszip/3.10.1/jszip.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/js/select2.full.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/chart.js@4.4.1/dist/chart.umd.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/flatpickr"></script>
<script src="https://cdn.jsdelivr.net/npm/flatpickr/dist/l10n/pt.js"></script>

<script>
jQuery(function($){
  /* =================== Utils =================== */
  const fmtBR = n => (Number(n)||0).toLocaleString('pt-BR',{style:'currency',currency:'BRL'});
  const br2f  = s => parseFloat(String(s||'').replace(/[^\d,-]/g,'').replace(/\./g,'').replace(',','.'))||0;
  const regexEscape = s => String(s).replace(/[.*+?^${}()|[\]\\]/g,'\\$&');

  /* =================== Período (Flatpickr + filtro DataTables) =================== */
  let rangeStart = null, rangeEnd = null; // Date (sem hora)

  function toMidnight(d){
    if (!d) return null;
    return new Date(Date.UTC(d.getFullYear(), d.getMonth(), d.getDate()));
  }
  function saveRangeLS(){
    const obj = {
      start: rangeStart ? rangeStart.toISOString().slice(0,10) : '',
      end:   rangeEnd   ? rangeEnd.toISOString().slice(0,10)   : ''
    };
    localStorage.setItem('baixa_range', JSON.stringify(obj));
  }
  function loadRangeLS(){
    try{
      const raw = localStorage.getItem('baixa_range');
      if (!raw) return;
      const o = JSON.parse(raw||'{}');
      if (o.start) rangeStart = new Date(o.start + 'T00:00:00Z');
      if (o.end)   rangeEnd   = new Date(o.end   + 'T00:00:00Z');
    }catch(_){}
  }

  // DataTables custom search por data (usa <tr data-date="YYYY-MM-DD">)
  $.fn.dataTable.ext.search.push(function(settings, data, dataIndex){
    const tr = settings.aoData[dataIndex]?.nTr;
    const ds = tr ? tr.getAttribute('data-date') : null;
    if (!rangeStart && !rangeEnd) return true;
    if (!ds) return false;
    const d = new Date(ds + 'T00:00:00Z');
    if (rangeStart && d < rangeStart) return false;
    if (rangeEnd   && d > rangeEnd)   return false;
    return true;
  });

  const fp = flatpickr('#f-range', {
    mode: 'range',
    locale: flatpickr.l10ns.pt,
    dateFormat: 'd/m/Y',
    onClose: function(selectedDates){
      if (selectedDates.length === 2){
        rangeStart = toMidnight(selectedDates[0]);
        rangeEnd   = toMidnight(selectedDates[1]);
      } else if (selectedDates.length === 1){
        rangeStart = toMidnight(selectedDates[0]);
        rangeEnd   = toMidnight(selectedDates[0]);
      } else {
        rangeStart = rangeEnd = null;
      }
      saveRangeLS();
      table.draw(false);
    }
  });

  function setQuickRange(days){
    const today = new Date(); // local
    const end   = toMidnight(today);
    const start = toMidnight(new Date(today.getFullYear(), today.getMonth(), today.getDate() - (days-1)));
    rangeStart = start; rangeEnd = end;
    fp.setDate([start, end], true); // atualiza input e dispara onClose=true
    saveRangeLS();
    table.draw(false);
  }

  $('#r-7').on('click', ()=> setQuickRange(7));
  $('#r-30').on('click',()=> setQuickRange(30));
  $('#r-365').on('click',()=> setQuickRange(365));
  $('#r-clear').on('click',()=>{
    rangeStart = rangeEnd = null;
    fp.clear();
    saveRangeLS();
    table.draw(false);
  });

  // Restaura período salvo (se houver)
  loadRangeLS();
  if (rangeStart && rangeEnd){
    fp.setDate([rangeStart, rangeEnd], false);
  }

  /* =================== DataTable =================== */
  const table = $('#tFretes').DataTable({
    order: [[2,'asc']],
    dom: 'Bfrtip',
    responsive: true,
    pageLength: 50,
    buttons: [
      { extend:'excel', text:'Exportar Excel' },
      { extend:'print', text:'Imprimir' }
    ],
    language:{ url:'https://cdn.datatables.net/plug-ins/1.13.8/i18n/pt-BR.json' },
    columnDefs: [
      { targets:[7,8,9], className:'text-end' },
      { targets:[0], orderable:false, searchable:false }
    ],
    drawCallback: function(){
      $('#chkAll').prop('checked', false);
      atualizaTotaisVisiveis();
      atualizaBotaoBaixa();
      if ($('#offGrafico').hasClass('show')) renderDebtChart();
    }
  });

  // ENTER na busca global ⇒ aplica, seleciona visíveis e abre modal com auto-preenchimento
  $('#tFretes_filter input').on('keydown', function(e){
    if (e.key !== 'Enter') return;
    e.preventDefault();
    let fired = false;
    table.one('draw.dt', function(){ if(!fired){ fired=true; selectAllVisible(true); reallyOpenModal(true, /*prefill*/true); } });
    table.search(this.value || '').draw(false);
    setTimeout(()=>{ if(!fired){ fired=true; selectAllVisible(true); reallyOpenModal(true, true); } }, 160);
  });

  /* =================== Select2 =================== */
  $('.select2').select2({ theme:'bootstrap-5', width:'100%', allowClear:true, placeholder:'Digite para pesquisar…', closeOnSelect:false });

  function uniqueFromColumn(colIdx){
    const data = table.column(colIdx).data().toArray();
    const set = new Set(); data.forEach(v => { const t=$('<div>').html(v).text().trim(); if (t) set.add(t); });
    return Array.from(set).sort((a,b)=> a.localeCompare(b,'pt-BR',{sensitivity:'base'}));
  }
  function fillSelect2($el, values){ $el.empty(); values.forEach(v => $el.append(new Option(v, v, false, false))); $el.trigger('change.select2'); }
  fillSelect2($('#f-rem'),  uniqueFromColumn(3));
  fillSelect2($('#f-dest'), uniqueFromColumn(4));
  fillSelect2($('#f-org'),  uniqueFromColumn(5));
  fillSelect2($('#f-dsti'), uniqueFromColumn(6));

  function regexFromVals(vals){ if (!vals || !vals.length) return ''; return '^(' + vals.map(regexEscape).join('|') + ')$'; }
  $('#f-rem').on('change', function(){ table.column(3).search(regexFromVals($(this).val()||[]), true, false).draw(); });
  $('#f-dest').on('change', function(){ table.column(4).search(regexFromVals($(this).val()||[]), true, false).draw(); });
  $('#f-org').on('change', function(){ table.column(5).search(regexFromVals($(this).val()||[]), true, false).draw(); });
  $('#f-dsti').on('change', function(){ table.column(6).search(regexFromVals($(this).val()||[]), true, false).draw(); });

  // ENTER com Select2 aberto ⇒ aplica filtros atuais, seleciona visíveis e abre modal com auto-preenchimento
  let lastKeyWasEnter = false;
  document.addEventListener('keydown', e => { if (e.key==='Enter') lastKeyWasEnter = true; }, true);
  document.addEventListener('keyup',   e => { if (e.key==='Enter') lastKeyWasEnter = false; }, true);
  $('.select2').on('select2:close', function(){
    if ($('.select2-container--open').length) return;
    if (lastKeyWasEnter){
      let fired=false;
      table.one('draw.dt', function(){ if(!fired){ fired=true; selectAllVisible(true); reallyOpenModal(true, true); } });
      table.draw(false);
      setTimeout(()=>{ if(!fired){ fired=true; selectAllVisible(true); reallyOpenModal(true, true); } }, 160);
    }
  });

  $('#btnClearFilters').on('click', function(){
    $('#f-rem,#f-dest,#f-org,#f-dsti').val(null).trigger('change');
    $('#tFretes_filter input').val('');
    table.search('').columns([3,4,5,6]).search('').draw();
    clearSelection();
  });

  /* =================== Seleção =================== */
  const selected = new Set();

  function clearSelection(){
    selected.clear();
    $('#tFretes .rowchk').prop('checked', false);
    atualizaTotaisSelecionados();
    atualizaBotaoBaixa();
  }
  function selectAllVisible(checked){
    table.rows({ search: 'applied' }).every(function () {
      const $row = $(this.node()); const id = Number($row.data('id'));
      if (checked) selected.add(id); else selected.delete(id);
      $row.find('.rowchk').prop('checked', checked);
    });
    atualizaTotaisSelecionados(); atualizaBotaoBaixa();
  }
  $('#chkAll').on('change', function(){ selectAllVisible($(this).is(':checked')); });

  $('#tFretes tbody').on('change', '.rowchk', function(){
    const $row = $(this).closest('tr');
    const id = Number($row.data('id'));
    if ($(this).is(':checked')) selected.add(id); else selected.delete(id);
    atualizaTotaisSelecionados(); atualizaBotaoBaixa();
  });

  function atualizaTotaisVisiveis(){
    let fat=0, pag=0, dev=0;
    table.rows({search:'applied'}).every(function(){
      const d = this.data();
      fat += br2f($('<div>').html(d[7]).text());
      pag += br2f($('<div>').html(d[8]).text());
      dev += br2f($('<div>').html(d[9]).text());
    });
    $('#tv-dev').text(fmtBR(dev));
    $(table.column(7).footer()).text(fmtBR(fat));
    $(table.column(8).footer()).text(fmtBR(pag));
    $(table.column(9).footer()).text(fmtBR(dev));
  }
  function atualizaTotaisSelecionados(){
    const ids = Array.from(selected); let dev=0;
    table.rows().every(function(){
      const node = $(this.node()); const id = Number(node.data('id'));
      if (ids.includes(id)){ dev += br2f(node.find('td').eq(9).text()); }
    });
    $('#ts-dev').text(fmtBR(dev));
    $('#sel_dev_modal').text(fmtBR(dev));
    return dev;
  }
  function atualizaBotaoBaixa(){ $('#btnAbrirBaixa').prop('disabled', !(selected.size>0)); }

  /* =================== Modal (pagamento) =================== */
  function maskMoney(el){
    let v = (el.value||'').replace(/\D/g,'');
    el.value = v ? ((parseInt(v,10)/100).toFixed(2).replace('.',',').replace(/(\d)(?=(\d{3})+,)/g,'$1.')) : '';
    atualizaResumoModal();
  }
  $('.money-in').on('input', function(){ maskMoney(this); });

  function somaInformado(){
    return br2f($('#in_din').val()) + br2f($('#in_pix').val()) + br2f($('#in_deb').val()) + br2f($('#in_cre').val()) + br2f($('#in_des').val());
  }
  function atualizaResumoModal(){
    const dev = atualizaTotaisSelecionados();
    const inf = somaInformado();
    $('#inf_modal').text(fmtBR(inf));
    const falt = Math.max(0, dev - inf);
    $('#falt_modal').text(fmtBR(falt))
      .toggleClass('text-success', Math.abs(dev-inf)<0.01)
      .toggleClass('text-danger', (dev-inf)>0.01);
    $('#btnConfirm').prop('disabled', !(selected.size>0 && inf>0 && inf<=dev+0.001));
  }

  function prefillDinheiroComDevedor(){
    const dev = atualizaTotaisSelecionados();
    $('#in_din').val( dev ? dev.toFixed(2).replace('.',',') : '' );
    $('#in_pix,#in_deb,#in_cre,#in_des').val('');
    atualizaResumoModal();
  }

  $('#modalBaixa').on('show.bs.modal', function(){
    $('#sel_ids').val(Array.from(selected).join(','));
    atualizaTotaisSelecionados();
    atualizaResumoModal();
  });
  // Foco inicial: DESCONTO; depois DINHEIRO ao Enter
  $('#modalBaixa').on('shown.bs.modal', function(){ $('#in_des').trigger('focus').select(); });

  $('#btnQuitarDin').on('click', function(){
    prefillDinheiroComDevedor();
    $('#in_des').trigger('focus').select();
  });
  $('#btnQuitarPix').on('click', function(){
    const dev = atualizaTotaisSelecionados();
    $('#in_pix').val( dev ? dev.toFixed(2).replace('.',',') : '' );
    $('#in_din,#in_deb,#in_cre,#in_des').val('');
    atualizaResumoModal(); $('#in_des').trigger('focus').select();
  });
  $('#btnZerar').on('click', function(){
    $('#in_din,#in_pix,#in_deb,#in_cre,#in_des').val('');
    atualizaResumoModal(); $('#in_des').trigger('focus').select();
  });

  // Enter navegação dentro do modal
  $('.money-in').on('keydown', function (e) {
    if (e.key !== 'Enter') return;
    e.preventDefault();
    const confirmEnabled = !$('#btnConfirm').prop('disabled');

    if (this.id === 'in_des') {
      $('#in_din').trigger('focus').select();        // Desconto → Dinheiro
    } else if (this.id === 'in_din' && confirmEnabled) {
      $('#btnConfirm').trigger('click');             // Dinheiro → Confirmar (se válido)
    } else {
      $('#in_din').trigger('focus').select();        // Demais campos → Dinheiro
    }
  });

  // Abre modal programaticamente
  function reallyOpenModal(clean = true, prefill = false){
    if (selected.size === 0) selectAllVisible(true);
    if (table.rows({search:'applied'}).count() === 0){
      alert('Nenhum frete visível/selecionado para abrir pagamento.');
      return;
    }
    if (clean) $('#in_din,#in_pix,#in_deb,#in_cre,#in_des').val('');
    $('#sel_ids').val(Array.from(selected).join(','));
    atualizaTotaisSelecionados();
    if (prefill) prefillDinheiroComDevedor(); else atualizaResumoModal();

    const modalEl = document.getElementById('modalBaixa');
    const btn     = document.getElementById('btnAbrirBaixa');

    if (btn){ btn.disabled = false; btn.click(); }
    setTimeout(() => {
      if (modalEl && !modalEl.classList.contains('show')){
        try { (window.bootstrap||bootstrap).Modal.getOrCreateInstance(modalEl).show(); } catch(_) {}
      }
      setTimeout(() => { $('#in_des').trigger('focus').select(); }, 80);
    }, 20);
  }

  /* =================== Gráfico (Opcional) =================== */
  let debtChart = null;
  function buildDatasetFromTable(){
    const groupCol = parseInt($('#chart-group').val(), 10) || 4;
    const map = new Map();
    table.rows({search:'applied'}).every(function(){
      const d = this.data();
      const label = $('<div>').html(d[groupCol]).text().trim() || '(sem nome)';
      const dev   = br2f($('<div>').html(d[9]).text());
      if (dev <= 0) return;
      map.set(label, (map.get(label)||0) + dev);
    });
    const arr = Array.from(map.entries()).sort((a,b)=> b[1]-a[1]);
    const topN = Math.max(3, Math.min(30, parseInt($('#chart-topn').val(),10) || 10));
    const top = arr.slice(0, topN);
    const rest = arr.slice(topN);
    const restSum = rest.reduce((s, x)=> s + x[1], 0);
    if (restSum > 0) top.push(['Outros', restSum]);
    return { labels: top.map(x=> x[0]), data: top.map(x=> +x[1].toFixed(2)) };
  }
  function renderDebtChart(){
    const ctxNode = document.getElementById('debtChart'); if (!ctxNode) return;
    const type = $('#chart-type').val();
    const ds = buildDatasetFromTable();
    const ctx = ctxNode.getContext('2d');
    if (debtChart) { debtChart.destroy(); debtChart = null; }
    const commonOpts = {
      responsive: true, maintainAspectRatio: false,
      plugins: {
        legend: { position: 'bottom' },
        tooltip: { callbacks: { label: (ctx) => {
          const v = ctx.parsed; const soma = ds.data.reduce((a,b)=>a+b,0) || 1;
          const pct = (v/soma)*100;
          return `${ctx.label}: ${v.toLocaleString('pt-BR',{style:'currency',currency:'BRL'})} (${pct.toFixed(1)}%)`;
        } } },
        title: { display: true, text: 'Top devedores (linhas visíveis)' }
      }
    };
    const cfg = (type==='bar')
      ? { type:'bar', data:{ labels: ds.labels, datasets:[{ label: 'Devedor', data: ds.data }] },
          options: Object.assign({}, commonOpts, { scales:{ y:{ beginAtZero:true, ticks:{ callback:v=> v.toLocaleString('pt-BR',{style:'currency',currency:'BRL'}) } } } }) }
      : { type:'doughnut', data:{ labels: ds.labels, datasets:[{ label: 'Devedor', data: ds.data }] }, options: commonOpts };
    debtChart = new Chart(ctx, cfg); setTimeout(()=> { if (debtChart) debtChart.resize(); }, 50);
  }
  $('#chart-group, #chart-type, #chart-topn').on('change input', renderDebtChart);
  const off = document.getElementById('offGrafico');
  if (off){
    off.addEventListener('shown.bs.offcanvas', function(){ renderDebtChart(); });
    off.addEventListener('hidden.bs.offcanvas', function(){ if (debtChart) { debtChart.destroy(); debtChart = null; } });
  }

  /* =================== Spotlight (F3/F5/F6 + F4 abre, ESC aplica e fecha) =================== */
  const spotlight = document.getElementById('kb-spotlight');
  const spInput   = document.getElementById('kb-input');
  const spScope   = document.getElementById('kb-scope-name');
  const spList    = document.getElementById('kb-list');

  // chips container
  let spPicksBox = document.getElementById('kb-picks');
  if (!spPicksBox) {
    spPicksBox = document.createElement('div');
    spPicksBox.id = 'kb-picks';
    spPicksBox.style.marginTop = '8px';
    spPicksBox.style.display = 'flex';
    spPicksBox.style.flexWrap = 'wrap';
    spInput.parentNode.insertBefore(spPicksBox, spList);
  }

  const COL = { REM:3, DEST:4, ORIG:5, DESTI:6 };
  let currentScope = { col: COL.DEST, name: 'Destinatário' };

  const allValuesCache = { 3:[], 4:[], 5:[], 6:[] };
  function valuesFor(col){
    if (!allValuesCache[col].length){
      const set = new Set();
      table.column(col).data().toArray().forEach(v=>{ const t = $('<div>').html(v).text().trim(); if (t) set.add(t); });
      allValuesCache[col] = Array.from(set).sort((a,b)=> a.localeCompare(b,'pt-BR',{sensitivity:'base'}));
    }
    return allValuesCache[col];
  }

  let spAll = [], spFiltered = [], spIdx = -1;
  let spPicks = [];
  const norm = s => String(s||'').normalize('NFD').replace(/[\u0300-\u036f]/g,'').toLowerCase();

  function drawList(){
    spList.innerHTML = '';
    spFiltered.slice(0, 200).forEach((v,i)=>{
      const li = document.createElement('li');
      li.textContent = v; li.dataset.v = v;
      if (i === spIdx) li.classList.add('active');
      spList.appendChild(li);
    });
    if (spFiltered.length && spIdx < 0){ spIdx = 0; if (spList.children[0]) spList.children[0].classList.add('active'); }
  }
  function drawPicks(){
    spPicksBox.innerHTML = '';
    spPicks.forEach((v,idx)=>{
      const chip = document.createElement('span');
      chip.className = 'badge bg-secondary me-1 mb-1';
      chip.style.cursor = 'pointer';
      chip.innerHTML = v + ' &times;';
      chip.title = 'Remover';
      chip.addEventListener('click', ()=>{ spPicks.splice(idx,1); drawPicks(); });
      spPicksBox.appendChild(chip);
    });
  }
  function renderSpotlightList(q){
    spAll = valuesFor(currentScope.col);
    const nq = norm(q);
    spFiltered = nq ? spAll.filter(v => norm(v).includes(nq)) : spAll.slice(0, 200);
    if (!spFiltered.length && q) spFiltered = [q];
    spIdx = spFiltered.length ? 0 : -1; drawList();
  }

  function openSpotlight(col, name){
    currentScope = { col, name };
    spScope.textContent = name;
    spInput.value = '';
    spPicks = []; drawPicks();
    spotlight.style.display = 'block';
    renderSpotlightList('');
    setTimeout(()=> spInput.focus(), 10);
  }
  function closeSpotlight(){ spotlight.style.display = 'none'; spInput.blur(); spPicks = []; drawPicks(); }

  function addSpotlightPick(val){
    const picked = (val||'').trim() || spInput.value.trim();
    if (!picked) return;
    if (!spPicks.includes(picked)) spPicks.push(picked);
    drawPicks();
    spInput.value = '';
    renderSpotlightList('');
  }

  function applyPicksAndOpen(){
    if (!spPicks.length){
      const fallback = (spFiltered.length && spIdx>=0) ? spFiltered[spIdx] : spInput.value.trim();
      if (fallback) spPicks.push(fallback);
    }
    const rx = spPicks.length ? '^(' + spPicks.map(regexEscape).join('|') + ')$' : '';

    let fired = false;
    function proceed(){
      if (fired) return; fired = true;
      selectAllVisible(true);
      closeSpotlight();
      reallyOpenModal(true, /*prefill*/true);
    }
    table.one('draw.dt', proceed);
    table.column(currentScope.col).search(rx, true, false).draw(false);
    setTimeout(()=>{ if (!fired) proceed(); }, 160);
  }

  // NOVO: aplicar picks e FECHAR sem abrir modal (para usar F4 depois)
  function applyPicksNoOpen(){
    if (!spPicks.length){
      const fallback = (spFiltered.length && spIdx>=0) ? spFiltered[spIdx] : spInput.value.trim();
      if (fallback) spPicks.push(fallback);
    }
    const rx = spPicks.length ? '^(' + spPicks.map(regexEscape).join('|') + ')$' : '';

    let fired = false;
    function proceed(){
      if (fired) return; fired = true;
      selectAllVisible(true); // mantém os visíveis selecionados
      closeSpotlight();
    }
    table.one('draw.dt', proceed);
    table.column(currentScope.col).search(rx, true, false).draw(false);
    setTimeout(()=>{ if (!fired) proceed(); }, 160);
  }

  // mouse na lista → adiciona chip
  $('#kb-list').on('click','li', function(){ addSpotlightPick(this.dataset.v || this.textContent); });
  spInput.addEventListener('input', ()=> renderSpotlightList(spInput.value));

  /* =================== Teclado Global =================== */
  function isF4(ev){ return ev.key === 'F4' || ev.keyCode === 115; }

  document.addEventListener('keydown', function(e){
    // Spotlight aberto
    if (spotlight.style.display === 'block'){
      if (e.key === 'Escape'){ e.preventDefault(); applyPicksNoOpen(); return; } // ESC aplica & fecha
      if (e.key === 'ArrowDown'){ e.preventDefault(); if (spFiltered.length){ spIdx=(spIdx+1)%spFiltered.length; drawList(); const li=spList.children[spIdx]; if(li) li.scrollIntoView({block:'nearest'}); } return; }
      if (e.key === 'ArrowUp'){ e.preventDefault(); if (spFiltered.length){ spIdx=(spIdx-1+spFiltered.length)%spFiltered.length; drawList(); const li=spList.children[spIdx]; if(li) li.scrollIntoView({block:'nearest'}); } return; }
      if (e.key === 'Enter'){ e.preventDefault(); const choice = spFiltered.length && spIdx >= 0 ? spFiltered[spIdx] : spInput.value.trim(); addSpotlightPick(choice); return; }
      if (e.key === 'Backspace' && !spInput.value){ e.preventDefault(); spPicks.pop(); drawPicks(); return; }
      if (isF4(e)){ e.preventDefault(); applyPicksAndOpen(); return; }
      return; // bloqueia outras teclas enquanto aberto
    }

    // Atalhos globais
    switch (e.key) {
      case 'F3': e.preventDefault(); openSpotlight(4, 'Destinatário'); break;
      case 'F4': e.preventDefault();
        if (selected.size === 0) selectAllVisible(true);
        reallyOpenModal(true, /*prefill*/true);
        break;
      case 'F5': e.preventDefault(); openSpotlight(5, 'Origem'); break;
      case 'F6': e.preventDefault(); openSpotlight(6, 'Destino'); break;
      case 'F7': e.preventDefault(); selectAllVisible(true); break;
      case 'F8': e.preventDefault();
        if (selected.size===0) selectAllVisible(true);
        reallyOpenModal(false, /*prefill*/false);
        break;
      case 'F9': e.preventDefault();
        $('#f-rem,#f-dest,#f-org,#f-dsti').val(null).trigger('change');
        $('#tFretes_filter input').val(''); table.search('').columns([3,4,5,6]).search('').draw();
        clearSelection();
        break;
      case 'Escape':
        $('#f-rem,#f-dest,#f-org,#f-dsti').val(null).trigger('change');
        $('#tFretes_filter input').val(''); table.search('').columns([3,4,5,6]).search('').draw();
        clearSelection();
        try { (window.bootstrap||bootstrap).Modal.getOrCreateInstance(document.getElementById('modalBaixa')).hide(); } catch(_) {}
        // Spotlight fechado aqui não faz nada
        break;
      default: break;
    }
  }, true);

  // F4 "no-capture" (fallback, caso outro script intercepte)
  document.addEventListener('keydown', function(ev){
    const spotlightOpen = (spotlight.style.display === 'block');
    if (!spotlightOpen && isF4(ev)){
      ev.preventDefault();
      ev.stopPropagation();
      if (selected.size === 0) selectAllVisible(true);
      reallyOpenModal(true, /*prefill*/true);
    }
  }, true);
});
</script>


<?php
get_footer(); // chama o footer.php do tema
