<?php
/* 
Template Name: Página Simples
*/
get_header(); // chama o header.php do tema
?>


<?php
/**
 * Template Name: Cadastro de Encomenda — Contatos Locais (Com Debug)
 */
if (!defined('ABSPATH')) exit;

/* ==========================================================
 * CONTATOS: Remetentes e Destinatários (sem cache)
 *  - Dedup por nome apenas
 *  - Dois registros por post (quando houver): remetente e destinatário
 *  - Encomendas: filtra tipocat ∈ {Encomenda, Encomendas, Encomedass}
 * ========================================================== */
if (!function_exists('brq_get_all_contacts_for_autocomplete')) {
  function brq_get_all_contacts_for_autocomplete() {

    // helpers
    $norm = function($s){
      $s = trim((string)$s);
      if ($s === '') return '';
      if (function_exists('transliterator_transliterate')) {
        $s = transliterator_transliterate('NFD; [:Nonspacing Mark:] Remove; NFC', $s);
      } elseif (function_exists('iconv')) {
        $s = @iconv('UTF-8','ASCII//TRANSLIT//IGNORE',$s);
      }
      return mb_strtolower($s, 'UTF-8');
    };
    $digits = function($s){ return preg_replace('/\D+/', '', (string)$s); };
    $first_meta = function($pid, $keys){
      foreach ($keys as $k) {
        $v = get_post_meta($pid, $k, true);
        if (isset($v) && trim((string)$v) !== '') return $v;
      }
      return '';
    };
    $score = function($telD, $docD){
      $sc = 0;
      $lenDoc = strlen($docD);
      if ($lenDoc >= 14)      $sc += 4;  // CNPJ
      elseif ($lenDoc >= 11)  $sc += 3;  // CPF
      elseif ($lenDoc > 0)    $sc += 2;
      $lenTel = strlen($telD);
      if     ($lenTel >= 11)  $sc += 2;
      elseif ($lenTel >= 10)  $sc += 1;
      elseif ($lenTel > 0)    $sc += 1;
      return $sc;
    };

    $outByName = array(); // chave: nome normalizado | valor: row com _score

    // Quais CPTs buscar
    $post_types = array('clientes_encomendas', 'encomendas');

    foreach ($post_types as $pt) {
      if (!post_type_exists($pt)) continue;

      // args base
      $args = array(
        'post_type'              => $pt,
        'posts_per_page'         => 2000,
        'fields'                 => 'ids',
        'orderby'                => 'date',
        'order'                  => 'DESC',
        'no_found_rows'          => true,
        'update_post_meta_cache' => false,
        'update_post_term_cache' => false,
        'suppress_filters'       => true,
      );

      // filtra tipocat nas encomendas
      if ($pt === 'encomendas') {
        $args['meta_query'] = array(
          'relation' => 'OR',
          array('key' => 'tipocat', 'value' => 'Encomenda',   'compare' => '='),
          array('key' => 'tipocat', 'value' => 'Encomendas',  'compare' => '='),
          array('key' => 'tipocat', 'value' => 'Encomedass',  'compare' => '='),
        );
      }

      $ids = get_posts($args);

      foreach ($ids as $pid) {
        $post_title = (string) get_the_title($pid);

        /* ====== Remetente ====== */
        $nomeR = $first_meta($pid, array(
          'nome_remetente','remetente_nome','remetente','nomeRemetente'
        ));
        if ($nomeR === '') $nomeR = $post_title;

        $telR  = $first_meta($pid, array(
          'telefone_remetente','contato_remetente','celular_remetente','whatsapp_remetente',
          'telefoneRemetente','contatoRemetente','whatsappRemetente',
          'telefone','contato','phone','whatsapp'
        ));
        $docR  = $first_meta($pid, array(
          // cobre typo e variações
          'cpf_remetente','cpf_rementente','cpfRemetente',
          'documento_remetente','doc_remetente','cnpj_remetente',
          'cpf_cnpj','cpf','doc'
        ));

        $nomeR_trim = trim((string)$nomeR);
        if ($nomeR_trim !== '') {
          $key = $norm($nomeR_trim);
          $telD = $digits($telR);
          $docD = $digits($docR);
          $sc   = $score($telD, $docD);

          // se não existe ou se este tem score melhor, substitui
          if (!isset($outByName[$key]) || $sc > $outByName[$key]['_score']) {
            $outByName[$key] = array(
              'id'       => (int)$pid,
              'nome'     => (string)$nomeR_trim,
              'telefone' => (string)$telR,
              'cpf'      => (string)$docR,
              '_score'   => $sc,
            );
          }
        }

        /* ====== Destinatário ====== */
        $nomeD = $first_meta($pid, array(
          'nome_destinatario','destinatario_nome','destinatario','nomeDestinatario'
        ));
        // só cria registro se realmente houver nome de destinatário
        if (trim((string)$nomeD) !== '') {
          $telD2 = $first_meta($pid, array(
            'telefone_destinatario','contato_destinatario','celular_destinatario','whatsapp_destinatario',
            'telefoneDestinatario','contatoDestinatario','whatsappDestinatario'
          ));
          $docD2 = $first_meta($pid, array(
            'cpf_destinatario','cpfDestinatario','documento_destinatario','doc_destinatario','cnpj_destinatario',
            'cpf_cnpj','cpf','doc'
          ));

          $key2 = $norm($nomeD);
          $telDigits = $digits($telD2);
          $docDigits = $digits($docD2);
          $sc2       = $score($telDigits, $docDigits);

          if (!isset($outByName[$key2]) || $sc2 > $outByName[$key2]['_score']) {
            $outByName[$key2] = array(
              'id'       => (int)$pid,
              'nome'     => (string)$nomeD,
              'telefone' => (string)$telD2,
              'cpf'      => (string)$docD2,
              '_score'   => $sc2,
            );
          }
        }
      } // foreach ids
    } // foreach post_types

    // monta saída final (removendo _score)
    $out = array();
    foreach ($outByName as $row) { unset($row['_score']); $out[] = $row; }

    error_log("BRQ_CONTATOS (sem cache) -> ".count($out)." contatos em ".date('Y-m-d H:i:s'));
    return $out;
  }
}

// Torna os contatos disponíveis ao JS
$ALL_CONTACTS = brq_get_all_contacts_for_autocomplete();
$BRQ_CONTATOS = $ALL_CONTACTS;

/* =========================
 * PRODUTOS do catálogo
 * ========================= */
if (!function_exists('brq_get_all_products_for_catalog')) {
  function brq_get_all_products_for_catalog() {
    $out = array();
    if (post_type_exists('produto_encomenda')) {
      $q = get_posts(array(
        'post_type'              => 'produto_encomenda',
        'posts_per_page'         => -1,
        'fields'                 => 'ids',
        'orderby'                => 'title',
        'order'                  => 'ASC',
        'no_found_rows'          => true,
        'update_post_meta_cache' => false,
        'update_post_term_cache' => false,
        'suppress_filters'       => true,
      ));
      foreach ($q as $pid) {
        $nome  = get_the_title($pid);
        if (trim((string)$nome) === '') continue;

        $valor = get_post_meta($pid, 'valor', true);
        if ($valor === '' || $valor === null) {
          $valor = get_post_meta($pid, 'preco', true);
        }

        $out[] = array(
          'id'    => (int)$pid,
          'nome'  => (string)$nome,
          'valor' => (float)($valor ?: 0),
        );
      }
    }
    return $out;
  }
}
$BRQ_PRODUTOS = brq_get_all_products_for_catalog();
?>


<!DOCTYPE html>
<html lang="pt-BR">
<head>
<meta charset="UTF-8" />
<meta name="viewport" content="width=device-width, initial-scale=1.0" />
<title>Cadastro de Encomenda</title>

 <link href="https://cdn.jsdelivr.net/npm/bootstrap-icons@1.10.5/font/bootstrap-icons.css" rel="stylesheet" />
<link rel="stylesheet" href="https://code.jquery.com/ui/1.12.1/themes/base/jquery-ui.css" />
<link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.0.0-beta3/css/all.min.css" />

<script src="https://cdnjs.cloudflare.com/ajax/libs/jquery/3.6.0/jquery.min.js"></script>
<script src="https://code.jquery.com/ui/1.12.1/jquery-ui.js"></script>
<script src="https://cdnjs.cloudflare.com/ajax/libs/jquery.mask/1.14.16/jquery.mask.min.js"></script>

<?php wp_head(); ?>

<script>
  // Dados carregados do PHP para o front
  window.BRQ_CONTATOS = <?php echo wp_json_encode($BRQ_CONTATOS,  JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES); ?>;
  window.BRQ_PRODUTOS = <?php echo wp_json_encode($BRQ_PRODUTOS,  JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES); ?>;

  // Ajax + Nonce
  window.ajaxurl = <?php echo json_encode(admin_url('admin-ajax.php')); ?>;
  window.addProdutoNonce = <?php echo json_encode(wp_create_nonce('brq_add_prod_nonce')); ?>;
</script>

<style>
  input:required:invalid { background-color:#ffe6e6; }
  input:required:valid   { background-color:#e6ffe6; }

  .ui-autocomplete{
    max-height:260px; overflow-y:auto; overflow-x:hidden;
    z-index:2000!important; font-size:0.9rem;
  }

  .form-label{ font-size:0.9rem; }
  .form-control,.form-select{ font-size:0.85rem; padding:0.5rem; height:auto; }
  .form-select{ width:100%; }

  .box{ border:1px solid #ddd; padding:20px; border-radius:8px; background-color:#E3F2FD; margin-bottom:10px; }

  .form-container{ display:flex; flex-wrap:wrap; gap:15px; }
  .col-33{ flex:1; min-width:100%; }
  .col-50{ flex:1; min-width:100%; }

  .table{ width:100%; margin-bottom:1rem; border-collapse:collapse; }
  .table th,.table td{ padding:0.6rem; vertical-align:middle; border-top:1px solid #dee2e6; text-align:left; }
  .table thead th{ border-bottom:2px solid #dee2e6; }

  @media (min-width:992px){ .col-33{ min-width:30%; } .col-50{ min-width:48%; } }
  @media (max-width:768px){
    .box{ padding:15px; }
    .form-label{ font-size:0.8rem; }
    .form-control,.form-select{ font-size:0.8rem; padding:0.4rem; }
  }
</style>
</head>
<body>


<form id="encomendaForm" method="post" action="<?php echo esc_url(admin_url('admin-post.php')); ?>" class="container mt-3">
  <input type="hidden" name="action" value="cadastrar_encomenda" />
  <?php wp_nonce_field('cadastrar_encomenda','cad_encomenda_nonce'); ?>
  <input type="hidden" id="produtosSelecionados" name="produtos_selecionados" />
  <input type="hidden" name="tipocat" value="Encomenda" />

  <div class="form-container">



  
    <!-- Informações da Encomenda -->
    <div class="box col-33">
      <h6 class="mb-2">Informações da Encomenda</h6>
      <div class="row g-2 mb-2">
        <div class="col-md-12">
          <select class="form-select" name="tipoP" id="tipoP" required>
            <option value="Avista">Avista</option>
            <option value="Receber">A Receber</option>
            <option value="Faturado">Faturado</option>
          </select>
        </div>
      </div>
      <div class="row g-2 mb-2">
        <div class="col-md-6">
          <select class="form-select" name="origem" id="origem" required>
            <option value="Manaus">Manaus</option>
            <option value="Codajás">Codajás</option>
            <option value="Coari">Coari</option>
            <option value="Tefé">Tefé</option>
            <option value="Alvarães">Alvarães</option>
            <option value="Uarini">Uarini</option>
          </select>
        </div>
        <div class="col-md-6">
          <select class="form-select" name="destino" id="destino" required>
            <option value="Manaus">Manaus</option>
            <option value="Codajás">Codajás</option>
            <option value="Coari">Coari</option>
            <option value="Tefé">Tefé</option>
            <option value="Alvarães">Alvarães</option>
            <option value="Uarini">Uarini</option>
          </select>
        </div>
        <div class="col-md-12">
          <label for="Transporte" class="form-label">Transporte:</label>
          <select class="form-select" name="transporte" id="Transporte" required>
            <?php
            $args = array(
              'post_type' => 'empresa',
              'posts_per_page' => -1,
              'orderby' => 'title',
              'order' => 'DESC'
            );
            $empresa_query = new WP_Query($args);
            if ($empresa_query->have_posts()) :
              while ($empresa_query->have_posts()) : $empresa_query->the_post(); ?>
                <option value="<?php echo esc_attr(get_the_ID()); ?>"><?php echo esc_html(get_the_title()); ?></option>
              <?php endwhile; wp_reset_postdata();
            else : ?>
              <option value="">—</option>
            <?php endif; ?>
          </select>
        </div>
        <div class="col-md-12 mb-2" id="campo-status" style="display:none;">
          <label for="status" class="form-label">Status</label>
          <select id="status" name="status" class="form-control">
            <option value="Recebida">Recebida</option>
            <option value="Entregue">Entregue</option>
          </select>
        </div>
      </div>
    </div>

    <!-- Remetente -->
    <div class="box col-33">
      <h6 class="mb-2">Remetente</h6>
      <div class="mb-2">
        <input type="text" class="form-control" id="nomeRemetente" name="nome_remetente" placeholder="Digite o nome" autocomplete="off" required>
      </div>
      <div class="mb-2">
        <input type="text" class="form-control telefone" id="contatoRemetente" name="contato_remetente" placeholder="Digite o Telefone"  >
      </div>
      <div class="mb-2">
        <input type="text" class="form-control cpf-cnpj" id="cpfRemetente" name="cpf_remetente" placeholder="Informe o RG, CPF ou CNPJ">
      </div>
    </div>

    <!-- Destinatário -->
    <div class="box col-33">
      <h6 class="mb-2">Destinatário</h6>
      <div class="mb-2">
        <input type="text" class="form-control" id="nomeDestinatario" name="nome_destinatario" placeholder="Digite o nome" autocomplete="off" required>
      </div>
      <div class="mb-2">
        <input type="text" class="form-control telefone" id="contatoDestinatario" name="contato_destinatario" placeholder="Digite o Telefone">
      </div>
      <div class="mb-2">
        <input type="text" class="form-control cpf-cnpj" id="cpfDestinatario" name="cpf_destinatario" placeholder="Informe o RG, CPF ou CNPJ">
      </div>
    </div>
  </div>

  <!-- Seleção de Produtos & Pagamento -->
  <div class="form-container">
    <!-- Produtos -->
    <div class="box col-50">
      <h6 class="mb-2">Produtos</h6>

      <div class="accordion" id="accordionProdutos">
        <div class="accordion-item">
          <h2 class="accordion-header" id="headingAdicionarProduto">
            <button class="accordion-button" type="button" data-bs-toggle="collapse" data-bs-target="#collapseAdicionarProduto" aria-expanded="true" aria-controls="collapseAdicionarProduto">
              Adicionar produto manualmente
            </button>
          </h2>
          <div id="collapseAdicionarProduto" class="accordion-collapse collapse show" aria-labelledby="headingAdicionarProduto" data-bs-parent="#accordionProdutos">
            <div class="accordion-body">
              <div class="row g-3 align-items-end">
                <div class="col-md-4">
                  <label for="produto_manual_nome" class="form-label">Produto</label>
                  <input type="text" id="produto_manual_nome" name="produto_manual_nome" class="form-control" placeholder="Nome do produto">
                </div>
                <div class="col-md-3">
                  <label for="produto_manual_valor" class="form-label">Valor</label>
                  <input type="text" id="produto_manual_valor" name="produto_manual_valor" class="form-control dinheiro" placeholder="0,00">
                </div>
                <div class="col-md-2">
                  <label for="produto_manual_qtd" class="form-label">Qtd</label>
                  <input type="number" id="produto_manual_qtd" name="produto_manual_qtd" class="form-control" value="1" min="1">
                </div>
                <div class="col-md-3">
                  <button type="button" id="adicionar_produto_manual" class="btn btn-dark w-100">
                    <span id="spinner" class="spinner-border spinner-border-sm me-2" style="display:none" role="status"></span>
                    Adicionar
                  </button>
                </div>
              </div>
              <div id="mensagem_sucesso" class="mt-3 alert alert-success d-none" role="alert">
                <i class="bi bi-check-circle-fill"></i> Produto adicionado com sucesso!
              </div>
              <div id="mensagem_erro" class="mt-3 alert alert-danger d-none" role="alert">
                <i class="bi bi-exclamation-triangle-fill"></i> Erro ao adicionar o produto.
              </div>
            </div>
          </div>
        </div>
      </div>

      <hr>
      <input type="text" id="buscaProduto" class="form-control" placeholder="Digite para buscar produtos (via catálogo)">

      <div class="mt-3 table-responsive">
        <h6>Lista de Produtos Selecionados:</h6>
        <table class="table table-bordered">
          <thead>
            <tr>
              <th>Produto</th>
              <th style="width:90px">Qtd</th>
              <th style="width:120px">Valor</th>
              <th style="width:120px">Total</th>
              <th style="width:60px">X</th>
            </tr>
          </thead>
          <tbody id="listaProdutos"></tbody>
        </table>
      </div>

      <label for="obs" class="form-label">Observação dos Produtos</label>
      <textarea class="form-control" id="obs" name="obs" rows="3"></textarea>
    </div>

    <!-- Pagamento e Totais -->
    <div class="box col-50">
      <h6 class="mb-2">Pagamento e Totais</h6>
      <div class="row">
        <div class="col-md-4 mb-2">
          <label for="valorTotalProdutos" class="form-label">Valor Produtos (R$):</label>
          <input type="text" class="form-control" id="valorTotalProdutos" name="valor_total_produtos" placeholder="0,00" readonly>
        </div>
        <div class="col-md-4 mb-2">
          <label for="valorTotalPago" class="form-label">Valor Pago (R$):</label>
          <input type="text" class="form-control" id="valorTotalPago" name="valor_total_pago" placeholder="0,00" readonly>
        </div>
        <div class="col-md-4 mb-2">
          <label for="valorTotalDevido" class="form-label">Valor Devido (R$):</label>
          <input type="text" class="form-control" id="valorTotalDevido" name="valor_total_devido" placeholder="0,00" readonly>
        </div>
      </div>
      <div class="row">
        <div class="col-md-6 mb-2">
          <label for="dinheiro" class="form-label">Dinheiro (R$):</label>
          <input type="text" class="form-control valor-pagamento" id="dinheiro" name="valor_dinheiro" placeholder="0,00">
        </div>
        <div class="col-md-6 mb-2">
          <label for="pix" class="form-label">Pix (R$):</label>
          <input type="text" class="form-control valor-pagamento" id="pix" name="valor_pix" placeholder="0,00">
        </div>
        <div class="col-md-6 mb-2">
          <label for="debito" class="form-label">Débito (R$):</label>
          <input type="text" class="form-control valor-pagamento" id="debito" name="valor_debito" placeholder="0,00">
        </div>
        <div class="col-md-6 mb-2">
          <label for="credito" class="form-label">Crédito (R$):</label>
          <input type="text" class="form-control valor-pagamento" id="credito" name="valor_credito" placeholder="0,00">
        </div>
        <div class="col-md-12 mb-2">
          <label for="desconto" class="form-label">Desconto (R$):</label>
          <input type="text" class="form-control valor-pagamento" id="desconto" name="valor_desconto" placeholder="0,00">
        </div>
        <div class="col-12 mb-2">
          <div class="row g-2 align-items-center">
            <label for="setor" class="col-auto col-form-label">Setor</label>
            <div class="col">
              <select id="setor" name="setor" class="form-select form-select-sm shadow-sm w-100" required>
                <option value="" selected disabled hidden>Selecione…</option>
                <option value="Porão">Porão</option>
                <option value="Escritório">Escritório</option>
              </select>
            </div>
          </div>
        </div>
        <input type="hidden" id="data_pagamento" name="data_pagamento" />
      </div>
    </div>
  </div>

  <div class="text-center mt-4">
    <button id="enviarBtn" type="submit" class="btn btn-dark btn-lg">
      <i class="fas fa-paper-plane"></i> Enviar Encomenda
    </button>
  </div>
</form>

<script>
// ===== Utilidades de moeda (pt-BR) =====
function brToFloat(s){
  if (!s) return 0;
  return parseFloat(String(s).replace(/\./g,'').replace(',','.')) || 0;
}
function floatToBR(v){
  return (parseFloat(v)||0).toFixed(2).replace('.',',');
}
function formatMoneyInput($el){
  $el.on('input', function(){
    let v = this.value.replace(/\D/g,'');
    v = (parseInt(v||'0',10)/100).toFixed(2);
    this.value = floatToBR(v);
  });
}

jQuery(function($){
  // ====== CAMPOS E ESTADO ======
  const $tabela = $('#listaProdutos');
  const $tipoP = $('#tipoP');
  const $valorTotalProdutos = $('#valorTotalProdutos');
  const $valorTotalPago = $('#valorTotalPago');
  const $valorTotalDevido = $('#valorTotalDevido');
  const produtosSelecionados = {}; // { id: {nome,preco,quantidade} }

  // Máscaras
  $('.telefone').mask('(00) 0 0000-0000');
  $('.cpf-cnpj').on('input', function(){
    const n = this.value.replace(/\D/g,'');
    if (n.length <= 11) $(this).mask('000.000.000-00');
    else               $(this).mask('00.000.000/0000-00');
  }).trigger('input');

  ['#dinheiro','#pix','#debito','#credito','#desconto'].forEach(sel => formatMoneyInput($(sel)));

  // ====== LÓGICA DE TIPO DE PAGAMENTO ======
  function aplicarTipoPagamento(){
    const tipo = $tipoP.val();
    const $campos = $('[name="valor_dinheiro"], [name="valor_pix"], [name="valor_debito"], [name="valor_credito"], [name="valor_desconto"]');
    if (tipo === 'Avista'){
      const now = new Date();
      const dd = String(now.getDate()).padStart(2,'0');
      const mm = String(now.getMonth()+1).padStart(2,'0');
      const yyyy = now.getFullYear();
      const HH = String(now.getHours()).padStart(2,'0');
      const MM = String(now.getMinutes()).padStart(2,'0');
      $('#data_pagamento').val(`${dd}-${mm}-${yyyy} ${HH}:${MM}`);
      $campos.prop('disabled', false).removeClass('bg-light');
    } else if (tipo === 'Receber'){
      $('#data_pagamento').val('');
      $campos.prop('disabled', true).addClass('bg-light').val('');
    } else {
      $('#data_pagamento').val('');
      $campos.prop('disabled', false).removeClass('bg-light');
    }
    calcularTotais();
  }
  $tipoP.on('change', aplicarTipoPagamento);

  // ====== TOTAIS ======
  function recalcularTotalProdutos(){
    let total = 0;
    Object.values(produtosSelecionados).forEach(p => total += (p.preco * p.quantidade));
    $valorTotalProdutos.val(floatToBR(total));
  }
  function calcularTotais(){
    const totalProdutos = brToFloat($valorTotalProdutos.val());
    const desconto = brToFloat($('#desconto').val());
    const dinheiro = brToFloat($('#dinheiro').val());
    const pix = brToFloat($('#pix').val());
    const debito = brToFloat($('#debito').val());
    const credito = brToFloat($('#credito').val());
    const totalPago = dinheiro + pix + debito + credito;
    $valorTotalPago.val(floatToBR(totalPago));
    const devido = Math.max(totalProdutos - totalPago - desconto, 0);
    $valorTotalDevido.val(floatToBR(devido));
    if ($tipoP.val()==='Avista') {
      $('#enviarBtn').prop('disabled', totalPago !== (totalProdutos - desconto));
    } else {
      $('#enviarBtn').prop('disabled', false);
    }
  }
  $('.valor-pagamento, #desconto').on('input', calcularTotais);

  function atualizarCampoProdutosHidden(){
    $('#produtosSelecionados').val(JSON.stringify(produtosSelecionados));
  }
  function desenharLinha(id){
    const p = produtosSelecionados[id];
    const total = floatToBR(p.preco * p.quantidade);
    return `
      <tr data-produto-id="${id}">
        <td>${p.nome}</td>
        <td><input type="number" class="form-control qtd" value="${p.quantidade}" min="1" style="width:70px"></td>
        <td>R$ ${floatToBR(p.preco)}</td>
        <td class="total">R$ ${total}</td>
        <td><button class="btn btn-sm btn-danger remover" type="button">Remover</button></td>
      </tr>`;
  }
  function inserirOuAtualizarLinha(id){
    const $exist = $tabela.find(`tr[data-produto-id="${id}"]`);
    if ($exist.length) $exist.replaceWith(desenharLinha(id));
    else               $tabela.append(desenharLinha(id));
    atualizarCampoProdutosHidden();
    recalcularTotalProdutos();
    calcularTotais();
  }

  // ====== PRODUTO MANUAL (SALVA NO BANCO via AJAX) ======
  $('#adicionar_produto_manual').on('click', function(){
    const nome = String($('#produto_manual_nome').val()||'').trim();
    const valor = brToFloat($('#produto_manual_valor').val());
    const qtd = Math.max(1, parseInt($('#produto_manual_qtd').val()||'1',10));
    const $err = $('#mensagem_erro');
    const $ok = $('#mensagem_sucesso');
    const $spin = $('#spinner');

    if (!nome || !valor){
      $err.text('Preencha nome e valor.').removeClass('d-none');
      setTimeout(()=> $err.addClass('d-none'), 3000);
      return;
    }

    $spin.show();
    $.post(window.ajaxurl, {
      action: 'adicionar_produto_manual',
      nome: nome,
      valor: valor,
      nonce: window.addProdutoNonce // será validado se você ativou no functions.php
    }, function(resp){
      $spin.hide();
      if (resp && resp.success){
        const id = String(resp.data.id); // usa ID real do post
        produtosSelecionados[id] = {
          nome: resp.data.nome,
          preco: parseFloat(resp.data.valor) || 0,
          quantidade: qtd
        };
        inserirOuAtualizarLinha(id);

        // limpa campos
        $('#produto_manual_nome').val('');
        $('#produto_manual_valor').val('');
        $('#produto_manual_qtd').val('1');

        // adiciona ao catálogo local e reindexa busca
        if (Array.isArray(window.BRQ_PRODUTOS)) {
          window.BRQ_PRODUTOS.unshift({
            id: parseInt(resp.data.id,10),
            nome: resp.data.nome,
            valor: parseFloat(resp.data.valor)||0
          });
          if (typeof rebuildProdIndex === 'function') rebuildProdIndex();
        }

        $ok.text('Produto adicionado com sucesso!').removeClass('d-none');
        setTimeout(()=> $ok.addClass('d-none'), 3000);
      } else {
        $err.text((resp && resp.data && resp.data.message) || 'Erro ao adicionar produto.').removeClass('d-none');
        setTimeout(()=> $err.addClass('d-none'), 3000);
      }
    }).fail(function(){
      $spin.hide();
      $err.text('Falha na comunicação.').removeClass('d-none');
      setTimeout(()=> $err.addClass('d-none'), 3000);
    });
  });

  // ====== BUSCA DE PRODUTO LOCAL (Catálogo em memória) ======
  const normP = s => (s||'').toString()
    .normalize('NFD').replace(/\p{Diacritic}/gu,'')
    .toLowerCase().trim();

  let PRODUTOS = (Array.isArray(window.BRQ_PRODUTOS) ? window.BRQ_PRODUTOS : []).map(p => ({
    ...p, _n: normP(p.nome)
  }));

  window.rebuildProdIndex = function(){
    PRODUTOS = (Array.isArray(window.BRQ_PRODUTOS) ? window.BRQ_PRODUTOS : []).map(p => ({...p, _n: normP(p.nome)}));
  };

  const LIM_INICIAL = 500;
  const LIM_POR_BUSCA = 100;

  function filtrarProdutosLocal(termo){
    const t = normP(termo);
    if (!t) return PRODUTOS.slice(0, LIM_INICIAL);

    const begins = [], contains = [];
    for (const p of PRODUTOS){
      const starts = p._n.startsWith(t);
      const has    = p._n.includes(t);
      if (starts) begins.push(p);
      else if (has) contains.push(p);
      if (begins.length + contains.length >= LIM_POR_BUSCA) break;
    }
    return begins.concat(contains).slice(0, LIM_POR_BUSCA);
  }

  function prodToItem(p){
    return {
      label: `${p.nome} — R$ ${floatToBR(p.valor||0)}`,
      value: p.nome,
      id: String(p.id),
      nome: p.nome,
      preco: parseFloat(p.valor) || 0
    };
  }

  $('#buscaProduto').autocomplete({
    minLength: 0,
    delay: 0,
    source: function(request, response){
      const lista = filtrarProdutosLocal(request.term).map(prodToItem);
      response(lista);
    },
    select: function(e, ui){
      const prod = ui.item;
      const id = String(prod.id);
      if (!produtosSelecionados[id]) {
        produtosSelecionados[id] = { nome: prod.nome, preco: prod.preco, quantidade: 1 };
      } else {
        produtosSelecionados[id].quantidade += 1;
      }
      inserirOuAtualizarLinha(id);
      $(this).val('');
      return false;
    }
  }).on('focus', function(){
    $(this).autocomplete('search', this.value);
  });

  // ====== Eventos da tabela ======
  $tabela.on('input', 'input.qtd', function(){
    const $tr = $(this).closest('tr');
    const id = String($tr.data('produto-id'));
    let q = parseInt($(this).val(),10);
    if (!isFinite(q) || q < 1) q = 1;
    produtosSelecionados[id].quantidade = q;
    inserirOuAtualizarLinha(id);
  });
  $tabela.on('click', 'button.remover', function(){
    const id = String($(this).closest('tr').data('produto-id'));
    delete produtosSelecionados[id];
    $(this).closest('tr').remove();
    atualizarCampoProdutosHidden();
    recalcularTotalProdutos();
    calcularTotais();
  });

  // ====== AUTOCOMPLETE LOCAL: Remetente/Destinatário ======
  function norm(s){ return (s||'').toString().normalize('NFD').replace(/\p{Diacritic}/gu,'').toLowerCase().trim(); }
  const CONTATOS = (Array.isArray(window.BRQ_CONTATOS) ? window.BRQ_CONTATOS : []).map(c=>({
    ...c, _n: norm(c.nome), _t: norm(c.telefone), _d: norm(c.cpf)
  }));
  const LIMITE_INICIAL = 500;
  const LIMITE_BUSCA = 50;

  function filtrarContatos(termo){
    const t = norm(termo);
    if (!t) return CONTATOS.slice(0, LIMITE_INICIAL);
    const begins=[], contains=[];
    for (const c of CONTATOS){
      const starts = c._n.startsWith(t) || (c._t && c._t.startsWith(t)) || (c._d && c._d.startsWith(t));
      const has    = c._n.includes(t)   || (c._t && c._t.includes(t))   || (c._d && c._d.includes(t));
      if (starts) begins.push(c); else if (has) contains.push(c);
      if (begins.length + contains.length >= LIMITE_BUSCA) break;
    }
    return begins.concat(contains).slice(0, LIMITE_BUSCA);
  }
  function toItem(c){
    return {
      label: c.nome + (c.telefone? ' — '+c.telefone:'') + (c.cpf? ' — '+c.cpf:''),
      value: c.nome,
      id: c.id,
      telefone: c.telefone||'',
      cpf: c.cpf||''
    };
  }
  function fonteLocal(request, response){
    response(filtrarContatos(request.term).map(toItem));
  }

  $('#nomeRemetente, #nomeDestinatario').autocomplete({
    source: fonteLocal,
    minLength: 0,
    delay: 0,
  }).on('focus', function(){
    $(this).autocomplete('search', this.value);
  }).on('autocompleteselect', function(e, ui){
    if (this.id === 'nomeRemetente'){
      $('#contatoRemetente').val(ui.item.telefone);
      $('#cpfRemetente').val(ui.item.cpf).trigger('input');
    } else {
      $('#contatoDestinatario').val(ui.item.telefone);
      $('#cpfDestinatario').val(ui.item.cpf).trigger('input');
    }
  });

  // ====== Navegação por Enter ======
  const ordem = ['nome_remetente','contato_remetente','cpf_remetente','nome_destinatario','contato_destinatario','cpf_destinatario','buscaProduto'];
  function focusNextOrClick(current){
    const idx = ordem.indexOf(current);
    if (current === 'produto_manual_valor'){
      $('#adicionar_produto_manual').trigger('click');
      const prev = ordem[idx-1];
      const $prev = $(`[name="${prev}"], [id="${prev}"]`).first();
      setTimeout(()=> $prev.trigger('focus'), 120);
      return;
    }
    if (idx > -1 && idx < ordem.length-1){
      const next = ordem[idx+1];
      $(`[name="${next}"], [id="${next}"]`).first().trigger('focus');
    }
  }
  $(document).on('keydown', function(e){
    if (e.key === 'Enter'){
      const el = e.target; const name = el.getAttribute('name') || el.getAttribute('id');
      if (name && ordem.includes(name)){
        e.preventDefault();
        focusNextOrClick(name);
      }
    }
  });

  // ====== Select2 ======
  $('#vvv, #ccc').select2({ allowClear:true, placeholder:'Selecione…', width:'100%' });

  // Persistência simples (localStorage)
  ['tipoP','origem','destino'].forEach(id=>{
    const key = 'select2_'+id; const $el = $('#'+id);
    const saved = localStorage.getItem(key);
    if (saved && $el.find(`option[value="${saved}"]`).length){ $el.val(saved).trigger('change'); }
    $el.on('change', ()=> localStorage.setItem(key, $el.val()||''));
  });

  // ====== Envio: evitar duplo e exigir 1+ produto ======
  let jaEnviado = false;
  $('#encomendaForm').on('submit', function(e){
    if (jaEnviado){ e.preventDefault(); return; }
    if ($('#listaProdutos tr').length === 0){
      e.preventDefault();
      alert('⚠️ Nenhum produto foi adicionado na lista!');
      return;
    }
    atualizarCampoProdutosHidden();
    jaEnviado = true;
    $('#enviarBtn').prop('disabled', true).html('<i class="fas fa-spinner fa-spin"></i> Enviando...');
  });

  // Inicializações
  aplicarTipoPagamento();
  recalcularTotalProdutos();
  calcularTotais();
});
</script>

<script src="https://cdn.jsdelivr.net/npm/select2@4.1.0-rc.0/dist/js/select2.min.js"></script>
<script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.0-alpha1/dist/js/bootstrap.bundle.min.js"></script>
</body>
</html>
<?php wp_footer(); ?>
